
# Copyright 2003,2004 Frederick Dean

use strict;
#use diagnostics;

use FKong::db;
use FKong::cgi;
use Date::Format;
use Digest::MD5 qw(md5_base64);
use FKong::user;
use Apache::Connection;

package FKong::session;

my $sessionCookieName = 'featurekong_session';
my $cookiePath = '/';  # we should derive this

# The web server tells us the remote IP of our browser.
# This returns the remote IP as a single integer.
sub remote_ip
{
   my $remoteip = 0;
   if($FKong::r->connection->remote_ip() =~ /^(\d+)\.(\d+)\.(\d+)\.(\d+)$/) {
      $remoteip = (((($1*256)+$2)*256)+$3)*256+$4;
   };
   return $remoteip;
}

sub make_a_session_cookie
{
   my $remoteip = remote_ip();
   my $randata = time() . $$ . rand() . \@_ . $remoteip;
   if(-r "/dev/urandom" && open(RAN,"</dev/urandom")) {  # If we can read from /dev/urandom
      read(RAN,$randata,32,length $randata);  # append more random data.
      close(RAN);
   }
   my $hash = Digest::MD5::md5_base64($randata);  
   $hash =~ tr/+\//_-/;  # make chars url friendly 
   $FKong::cgi::cookie{$sessionCookieName} = $hash;  
   FKong::db::SendSQL("REPLACE session SET cookie = ".FKong::db::SqlQuote($hash).", remoteip = $remoteip,\n".
               "lastused = UNIX_TIMESTAMP(), userid = 0");
}

sub get_state
{
   #my($package, $filename, $line) = caller;
   #print STDERR "get_state(". join(",",@_) .") filename=$filename line=$line\n";

   my $dbh = FKong::db::SendSQL("SELECT name, value FROM more_session\nWHERE name IN (". 
                         join(",",map { FKong::db::SqlQuote($_) } @_) .")\n".
                        "AND cookie = ". FKong::db::SqlQuote($FKong::cgi::cookie{$sessionCookieName}));
   my %values;
   while(my($name,$value) = $dbh->fetchrow_array()) { $values{$name} = $value };
   return $values{$_[0]} if ! wantarray;
   return map { $values{$_} } @_;
}

sub set_state
{
   my($name,$value,$duration) = @_;
   $duration ||= 60*60*24*7;  # default of one week
   FKong::db::SendSQL("REPLACE more_session SET name = ". FKong::db::SqlQuote($name) .",\nvalue=". FKong::db::SqlQuote($value) .",\n".
               "del_time = UNIX_TIMESTAMP() + $duration,\ncookie = ". FKong::db::SqlQuote($FKong::cgi::cookie{$sessionCookieName}));
}

my %non_keepers = ( logout => 1, password => 1, login_username => 1);

# This routine will show the login form and exit.
sub login_form
{
   $FKong::cgi::keyword{'hidden'} = "";
   foreach my $name (keys %FKong::cgi::form) {
      next if $non_keepers{$name};
      $FKong::cgi::keyword{'hidden'} .= "<input type=hidden name=$name value=\"". FKong::cgi::value_quote($FKong::cgi::form{$name}) ."\"\n"; 
   };
   $FKong::cgi::keyword{'formurl'} = FKong::cgi::Uri("") || "index.html";
   FKong::cgi::print_expanded_template("template/login-template.html");
   goto DONE;
}

$FKong::url_func{'login.html'} = \&login_form;

# This is an external function
sub require_login
{
   login_form() if $FKong::cgi::keyword{'userid'} == 99;
}

our %privname = ( view_users => 1, edit_users => 2, 
                  view_other => 0x10, edit_other => 0x20, 
                  view_fields => 0x100, edit_fields => 0x200, 
                  view_log => 0x400, 
                  view_features => 0x800, edit_features => 0x1000, 
                  recv_appl => 0x2000,
                  delete_features => 0x10000 );
sub has_priv
{
   my($priv) = @_;
   die "internal error: priv name" unless $privname{$priv};
   return $FKong::cgi::keyword{'privs'} & $privname{$priv};
}
sub must_have_priv
{
   my($priv) = @_;
   return 1 if has_priv($priv);
   require_login();
   if(! has_priv($priv)) {
      FKong::Fatal("Sorry $FKong::cgi::keyword{'realname'}, you have insufficient privileges to see this page.","no priv $priv");
   }
}

sub isPasswordValid
{
   my($username,$password) = @_;
   my $dbh = FKong::db::SendSQL("SELECT cryptpassword, userid, disabledtext, email, realname, privs\n".
                         "FROM user\n".
                         "WHERE username = ".FKong::db::SqlQuote($username));
   my($cryptpassword,$userid,$disabledtext,$email,$realname,$privs) = $dbh->fetchrow_array();
   return undef if ! defined $cryptpassword;  # if non-existent user
   my $pwtype = substr($cryptpassword,0,3);
   if($pwtype eq "md5") {
      # The cryptpassword has three bytes of scheme, 32 bytes of salt, then 32 bytes of crypt password.
      my $salt = substr($cryptpassword,0,25);
      # We don't use crypt() because we want more than 8 chars of relevance in our passwords!
      my $hash = Digest::MD5::md5_base64($salt . $password);
      if(substr($cryptpassword,25) ne $hash) {  # if invalid password
         return undef;
      }
   } else {  # else we don't know the pwtype
      return undef;  # because we don't understand the crypt password cypher
   } 
   return($userid,$disabledtext,$email,$realname,$privs,$pwtype);
}

sub logout
{
   foreach (qw/username realname privs email/) { $FKong::cgi::keyword{$_} = "" };
   FKong::db::SendSQL("UPDATE session SET isvalid = 0, lastused = UNIX_TIMESTAMP(), remoteip = ". remote_ip() ."\n".  # logout old session
               "WHERE cookie = ". FKong::db::SqlQuote($FKong::cgi::cookie{$sessionCookieName}));
}

# This routine uses $FKong::cgi::form{'username'} and $FKong::cgi::form{'password'} and cookie()
# This routine is called when the form data contains a username or a password.
# We will abandon the old session, if any.
# Unlike isPasswordValid() this one updates the global variables  $FKong::session::user_id and %FKong::session::group
sub validate_login
{
   # logout();
   my $session_cookie = $FKong::cgi::cookie{$sessionCookieName};
   die "internal error" unless $session_cookie;
   my $username = $FKong::cgi::form{'login_username'};
   FKong::Fatal("Login failed.<p>\nYou must enter a username.<br>",'no username') if ! $username; 
   my $password = $FKong::cgi::form{'login_password'};
   FKong::Fatal("Login failed.<p>\nYou must enter a password.<br>",'no passwd') if ! $password; 
   my($userid,$disabledtext,$email,$realname,$privs) = isPasswordValid($username,$password);
   FKong::Fatal("Login Failed.<p>\nYour username or password was invalid.<br>",'bad passwd') if ! $userid; 
   FKong::db::SendSQL("REPLACE session SET isvalid = 1, userid = $userid, lastused = UNIX_TIMESTAMP(), cookie = ". FKong::db::SqlQuote($session_cookie));
   if($FKong::cgi::keyword{'formurl'} =~ /login.html$/i) {  # if they explicitly visited the login form
      FKong::cgi::redirect(FKong::cgi::Uri("/"));  # redirect to the root page
      goto DONE;
   };
   if($FKong::cgi::form{'exclusive'}) {
      FKong::db::SendSQL("UPDATE session SET isvalid = 0 WHERE userid = $userid AND cookie != ". FKong::db::SqlQuote($session_cookie));
   };
}

sub make_session_valid
{
   my($userid) = @_;
   return if ! $FKong::cgi::cookie{$sessionCookieName};
   FKong::db::SendSQL("REPLACE session SET isvalid = 1, userid = $userid, lastused = UNIX_TIMESTAMP(),".
                      " cookie = ". FKong::db::SqlQuote($FKong::cgi::cookie{$sessionCookieName}));
}

sub send_cookie
{
         $FKong::r->headers_out->set("Set-Cookie","$sessionCookieName=$FKong::cgi::cookie{$sessionCookieName}; path=". 
                                 $FKong::r->location ."; expires=Sun, 30-Jun-2029 00:00:00 GMT");
}

# This is the big external function.
sub Check
{
   $FKong::cgi::keyword{'privs'} = 0;  # default to no privileges
   if($FKong::cgi::form{'apply'}) {  # if they are applying for an account
      FKong::user::apply_for_login();
   } elsif($FKong::cgi::form{'send_pw'}) {  # if they are requesting a password be reset and sent
      FKong::user::send_password($FKong::cgi::form{'send_name'});
      FKong::cgi::print_expanded_template("template/sent-template.html"); 
      goto DONE;
   } elsif($FKong::cgi::form{'logout'}) {  # if they are logging out
      logout();
   }
   my $session_cookie = $FKong::cgi::cookie{$sessionCookieName};
   if(!$session_cookie) {  # if there was no cookie
      make_a_session_cookie();  # create one
      if($FKong::cgi::form{'login'}) {  # if they are trying to log in
         FKong::Fatal("Cookies are required to login.  Please allow cookies and try to log in again.",'no cookies');
      }
      send_cookie();
   } else { # else there was a cookie and we need to check it
      if($FKong::cgi::form{'login'}) {  # if they are trying to log in
         validate_login();
      };
      # We could check for remote address to match, but we won't.
      my $dbh = FKong::db::SendSQL("SELECT lastused, session.userid,username,realname,isvalid,disabledtext,privs,email,debug\n".
                            "FROM session LEFT JOIN user USING (userid)\n".
                            "WHERE session.cookie = ".FKong::db::SqlQuote($session_cookie) ." LIMIT 1");
      my($lastused,$userid,$username,$realname,$isvalid,$disabledtext,$privs,$email,$debug) = $dbh->fetchrow_array(); 
      if(defined($lastused) && $lastused < (time() - $FKong::config{'MaxIdleTime'})) {  # if session has timed out
         # Clean out idle sessions.
         # We could do this more frequently if we wanted to, but this seems frequent enough.
         FKong::db::SendSQL("DELETE FROM session WHERE lastused < UNIX_TIMESTAMP() - $FKong::config{'MaxIdleTime'}"); 
         my $dbh = FKong::db::SendSQL("SELECT more_session.cookie FROM more_session LEFT JOIN session USING (cookie)\n".
                               "WHERE session.cookie is NULL GROUP BY cookie LIMIT 1");  # fetch state with missing cookie entry
         while(my($cookie) = $dbh->fetchrow_array()) {  # for every obsolete state entry
            FKong::db::SendSQL("DELETE FROM more_session WHERE cookie = ". FKong::db::SqlQuote($cookie));  # delete the state entry
         }
         make_a_session_cookie();  # create one
         send_cookie();
      } elsif($isvalid) {  # else valid session matched
         FKong::Fatal("Your account has been disabled because...<br>\n". 
                            FKong::cgi::Linkify($disabledtext) ."<br>",'disabled account') if $disabledtext; 
         $FKong::cgi::keyword{'username'} = $username;
         $FKong::cgi::keyword{'realname'} = $realname;
         $FKong::cgi::keyword{'privs'} = $privs;
         $FKong::cgi::keyword{'email'} = $email;
         $FKong::cgi::keyword{'userid'} = $userid;
         if(!$privs) {  # if no admin privs
             FKong::cgi::print_expanded_template("template/unapproved-template.html"); 
             goto DONE;
         };
         FKong::db::SendSQL("UPDATE user SET lastTS = UNIX_TIMESTAMP() WHERE userid = $userid");
         if($FKong::cgi::form{'toggle_debug'}) {
            must_have_priv("view_other");
            $debug = ! $debug;
            FKong::db::SendSQL("UPDATE session SET debug = ". ($debug || "0") .
                             " WHERE cookie = ". FKong::db::SqlQuote($FKong::cgi::cookie{$sessionCookieName}));
         };
         if($debug && ! $FKong::debug) {
              $FKong::debug = "session";
         };
      }
      # Update cookie use time
      FKong::db::SendSQL("UPDATE session SET remoteip = ". remote_ip() .",\nlastused = UNIX_TIMESTAMP()\n".
                  "WHERE cookie = ".FKong::db::SqlQuote($FKong::cgi::cookie{$sessionCookieName}));
   }
   if(! $FKong::cgi::keyword{'userid'}) {  # if they are not logged in
      my $dbh = FKong::db::SendSQL("SELECT username, privs, realname, disabledtext FROM user WHERE userid = 99 LIMIT 1");
      ($FKong::cgi::keyword{'username'},$FKong::cgi::keyword{'privs'},
       $FKong::cgi::keyword{'realname'},my $dtext) = $dbh->fetchrow_array();
      FKong::Fatal("$dtext.<br>") if $dtext; 
      $FKong::cgi::keyword{'userid'} = 99;
   }
}

1;

__END__

This minimalist session tracker requires cookies. 
Sessions are stored in a MySQL database.  The table
is called session.

You are logged in if (and only if) $FKong::cgi::keyword{'username'} is set.




