
# Copyright 2003,2004 Frederick Dean

use strict;

use FKong::list;

package FKong::field;

sub commands { 
   my($fieldId) = @_;
   return FKong::cgi::Link("type.html",["fieldId=$fieldId"]) ."Type</a>\n" .
          FKong::cgi::Link("edit.html",["fieldId=$fieldId"]) ."Edit</a>\n" .
          FKong::cgi::Link("delete.html",["fieldId=$fieldId"]) ."Delete</a>\n";
}

my %typeNames = ( mult_choice => "Multiple Choice", integer => "Integer", hexint => 'Hex Integer', money => 'Money',
                  oneline => "One Line", multiline => "Multiple Lines", user => "User", unix_ts => 'Unix Timestamp',
                  bool => "Boolean", float => 'Float', constant => "Constant HTML");

sub yes_no { return $_[0] ? "yes" : '<font color="#888888">no</font>' }

sub blank_zero { return $_[0] ? $_[0] : '&nbsp;' }

sub type_name { return $typeNames{$_[0]} || 'error' }

my $tbl = FKong::list->new('field_def','field_def');
#           nick private         title  align default   sql          display      descript                       
$tbl->DefCol("p90",0,            "Commands",1,1,"field_def.fieldId",   \&commands, "");
$tbl->DefCol("p10",0,               "DB ID",1,0,"field_def.fieldId",        undef, "Internal database id for field_def table.");
$tbl->DefCol("p20",0,          "Field Name",1,1,"field_def.fieldName",      undef, "");
$tbl->DefCol("p91",0,               "Table",1,0,"field_def.tablesql",       undef, "The SQL table name");
$tbl->DefCol("p21",0,            "SQL Name",1,0,"field_def.sqlName",        undef, "The internal table column name");
$tbl->DefCol("p34",0,                "Type",1,1,"field_def.type",      \&type_name,"");
$tbl->DefCol("p30",0,             "Default",1,1,"field_def.deflt",          undef, "");
$tbl->DefCol("p93",0,            "New Form",1,1,"field_def.on_new_form", \&yes_no, "On new form");
$tbl->DefCol("p36",0,           "List View",1,1,"field_def.list_show",   \&yes_no, "If column shows on the default table list");
$tbl->DefCol("p37",0,        "Basic Search",1,1,"field_def.basic_search",\&yes_no, "If field shows on the basic search form");
$tbl->DefCol("p38",0,         "Adv. Search",1,1,"field_def.adv_search",  \&yes_no, "If field shows on the advanced search form");
$tbl->DefCol("p39",0,         "New Summary",1,1,"field_def.new_summary", \&yes_no, "Does this field appear in the new record summary blurbs");
$tbl->DefCol("p42",0,     "Comment Summary",1,1,"field_def.cmt_summary", \&yes_no, "Does this field appear in the new comment summary blurbs");
$tbl->DefCol("p40",0,      "Batch Editable",1,1,"field_def.batch",       \&yes_no, "Can this be edited with multiple record edit form");
$tbl->DefCol("p92",0,  "Reset on Duplicate",1,1,"field_def.rset_on_dup", \&yes_no, "Reset to default when duplicating");
$tbl->DefCol("p41",0,              "Unique",1,1,"field_def.unique_",     \&yes_no, "Must this field be unique");
$tbl->DefCol("p44",0,          "Sort Order",1,1,"field_def.sort",           undef, "For placement on the new/edit forms.");
$tbl->DefCol("p26",0,                 "Min",1,1,"field_def.min",            undef, "Numerical limit or text length");
$tbl->DefCol("p28",0,                 "Max",1,1,"field_def.max",            undef, "Numerical limit or text length");
$tbl->DefCol("p68",0,               "Width",1,1,"field_def.width",          undef, "For HTML size= or cols= attribute");
$tbl->DefCol("p69",0,              "Height",1,1,"field_def.height",         undef, "For HTML rows= attribute");
$tbl->DefCol("p94",0,          "Short Help",1,0,"field_def.short_help",     undef, "Shows on the new/edit/search forms");
$tbl->DefCol("p95",0,   "Short Help Length",1,1,"LENGTH(field_def.short_help)",undef, "");
$tbl->DefCol("p98",0,           "Long Help",1,0,"field_def.long_help",      undef, "");
$tbl->DefCol("p99",0,    "Long Help Length",1,0,"LENGTH(field_def.long_help)",undef, "");
#$tbl->DefCol("p50",0,       "Created By ID",1,0,"field_def.create_by",      undef, "");
$tbl->DefCol("p51",0,           "Create By",1,0,"COALESCE(creator.username,field_def.create_by)",undef, "Who created the field",
             "LEFT JOIN user AS creator ON (creator.userid = field_def.create_by)\n");
$tbl->DefCol("p52",0,         "Create Time",1,0,"field_def.createTS",\&FKong::list::unixTimestamp, "When the field was created");
#$tbl->DefCol("p53",0,      "Modified By Id",1,0,"field_def.mod_by",         undef, "");
$tbl->DefCol("p54",0,         "Modified By",1,0,"COALESCE(modify.username,field_def.mod_by)",undef, "Who last modified the field",
             "LEFT JOIN user AS modify ON (modify.userid = field_def.mod_by)\n");
$tbl->DefCol("p55",0,         "Modify Time",1,0,"field_def.modifyTS",\&FKong::list::unixTimestamp, "When the field was last modified");

sub keeper_parameters
{
   return [ FKong::list::column_params($tbl) ];    # list reference
}

$FKong::table_func{'field/list.html'} = \&show_table;
$FKong::table_func{'field/list.csv'} = $FKong::table_func{'field/list.html'};

sub show_table
{
   my($fktable) = @_;

   # check that they have permission 
   FKong::session::must_have_priv("view_fields");
   # check for configure button
   if($FKong::cgi::form{'update'}) {  # If the now button was pressed
      update($fktable);
      return FKong::cgi::redirect("list.html");
   }
   if($FKong::cgi::form{'create'}) {  # If the now button was pressed
      create($fktable);
      return FKong::cgi::redirect("list.html");
   }
   if($FKong::cgi::form{'delete'}) {  
      dirty_work_of_delete($$fktable{'tablesql'});
      return FKong::cgi::redirect("list.html");
   }
   if($FKong::cgi::form{'delete_choices'}) {  
      migrate_choices($$fktable{'tablesql'},$$fktable{'pkey'});
      return FKong::cgi::redirect("list.html");
   }
   if($FKong::cgi::form{'cancel'}) {  
      return FKong::cgi::redirect("list.html");
   }
   # column headings
   my $keepers = keeper_parameters();
   my($colsql,$ordersql,$joins) = $tbl->get_column_sql($keepers,"p44");
   # maximum number of records stuff
   my $maxRecords = $FKong::cgi::form{'max_records'} || 100;
   $maxRecords = 1000 if $maxRecords > 1000;
   # Filter by fieldId
   my $where = "WHERE field_def.tablesql = ". FKong::db::SqlQuote($$fktable{'tablesql'}) ."\n";
   # submit the query
   my $query = "SELECT $colsql\nFROM field_def\n".
               $joins .  $where .
               "GROUP BY fieldId\n".
               "ORDER BY $ordersql\nLIMIT $maxRecords";
   $tbl->print_template_with_table("template/fields-template.html",$query,$keepers);
}

$FKong::table_func{'field/delete.html'} = \&delete_field;

# This function shows the form to delete a field definition.
sub delete_field
{
   my($fktable) = @_;
   FKong::session::must_have_priv("edit_fields");
   field_keyword_details($fktable);
   foreach (qw/on_new_form list_show basic_search adv_search new_summary cmt_summary index/) {
      $FKong::cgi::keyword{$_} = ($FKong::cgi::keyword{$_} ? "yes" : "no");  # convert to yes/no
   };
   $FKong::cgi::keyword{'default'} = ($FKong::cgi::keyword{'default'} ? "yes" : "no") if $FKong::cgi::keyword{'typeName'} eq "Boolean"; 
   FKong::cgi::print_expanded_template("template/field_delete-template.html"); 
}

$FKong::table_func{'field/edit.html'} = \&edit_field;

# This function shows the form to edit a field definition.
sub edit_field
{
   my($fktable) = @_;
   FKong::session::must_have_priv("view_fields"); # The edit form doubles as the detail page
   field_keyword_details($fktable);
   my $template = "$FKong::htmldir/template/field_edit_$FKong::cgi::keyword{'type'}-template.html";
   if(! -e $template) {
      $template = "template/field_edit-template.html"; 
   };
   FKong::cgi::print_expanded_template($template); 
} 

# This form sets the keywords representing the index status of a field.
# The data cannot be found in the field_def table.
sub index_selects
{
   my($fktable,$fieldId,$index2,$index3) = @_;
   my(@sqlNames) =   ('', $$fktable{'pkey'}, 'createTS',   'create_by', 'modifyTS',   'modify_by');
   my(@fieldNames) = ('(none)',"Record Id",  'Create Time','Created By','Modify Time','Modify By');
   my $dbh = FKong::db::SendSQL("SELECT fieldName,sqlName FROM field_def WHERE fieldId != ". FKong::db::SqlQuote($fieldId) ."\n".
                         "AND type IN ('oneline','bool','mult_choice','integer','hexint','money','float','user')\n".
                         "AND tablesql = ". FKong::db::SqlQuote($$fktable{'tablesql'}) ."\n".
                         "ORDER BY sort, fieldName");
   while(my($fieldName,$sqlName) = $dbh->fetchrow_array()) { 
      push(@fieldNames,$fieldName);
      push(@sqlNames,$sqlName);
   };
   my $index2Select = "<select name=index2>\n";
   my $index3Select = "<select name=index3>\n";
   foreach my $i (0..$#sqlNames) {
      my $selected2 = defined $index2 && $index2 eq $sqlNames[$i] ? " SELECTED" : "";
      my $selected3 = defined $index3 && $index3 eq $sqlNames[$i] ? " SELECTED" : "";
      $index2Select .= " <option value=\"$sqlNames[$i]\"$selected2>". FKong::cgi::htmlQuote($fieldNames[$i]) ."</option>\n";
      $index3Select .= " <option value=\"$sqlNames[$i]\"$selected3>". FKong::cgi::htmlQuote($fieldNames[$i]) ."</option>\n";
   }
   $FKong::cgi::keyword{'index2Select'} = "$index2Select</select>\n";
   $FKong::cgi::keyword{'index3Select'} = "$index3Select</select>\n";
}

# The return value is a list.  The first one is the key name.  The second two are the 2nd and 3rd index column names.
# The information comes from the actual database schema.
sub determine_current_index
{
   my($sqlName,$tablesql) = @_;
   my($found_key_name,$col2,$col3) = ("","","");
   my $dbh = FKong::db::SendSQL("SHOW INDEX FROM $tablesql");
   while(my($table,$unique,$key_name,$seq_in_index,$column_name) = $dbh->fetchrow_array()) {
      #print STDERR "sqlName=$sqlName key_name=$key_name seq_in_index=$seq_in_index column_name=$column_name\n";
      $found_key_name = $key_name if $seq_in_index == 1  && $column_name eq $sqlName;
      $col2 = $column_name if $key_name eq $found_key_name && $seq_in_index == 2;
      $col3 = $column_name if $key_name eq $found_key_name && $seq_in_index == 3;
   };
   #print STDERR "found_key_name=$found_key_name col2=$col2 col3=$col3\n";
   return($found_key_name,$col2,$col3);
}

# If the user is going to delete or update a field, this function initializes the
# keywords which those forms require.
sub field_keyword_details
{
   my($fktable) = @_;
   my $fieldId = $FKong::cgi::form{'fieldId'};
   $fieldId or FKong::Fatal("Internal error: fieldId missing.",'');
   $fieldId =~ /^\d+$/ or FKong::Fatal("Internal error: fieldId malformed.",'');
   my $dbh = FKong::db::SendSQL("SELECT  fieldName, field_def.tablesql, sqlName, type, min, max, deflt, on_new_form, list_show, basic_search,\n".
               "adv_search, new_summary, cmt_summary, batch, rset_on_dup, unique_, \n".
               "creator.username, field_def.createTS, modified.username,\n".
               "field_def.modifyTS, sort, field_def.short_help, long_help, width, height \n".
               "FROM field_def\n".
               "LEFT JOIN user AS creator ON (creator.userid = field_def.create_by)\n".
               "LEFT JOIN user AS modified ON (creator.userid = field_def.mod_by)\n".
               "WHERE fieldId = $fieldId AND field_def.tablesql = ". FKong::db::SqlQuote($$fktable{'tablesql'}) ."\n".
               "LIMIT 1");
   my($fieldName,$tablesql,$sqlName,$type,$min,$max,$default, $on_new_form,$list_show,$basic_search,
      $adv_search,$new_summary,$cmt_summary,$batch,$rset_on_dup,$unique,$create_by,$createTS,
      $mod_by,$modifyTS,$sort,$short_help,$long_help,$width,$height) = $dbh->fetchrow_array();
   $fieldName or FKong::Fatal("fieldId $fieldId does not exist.",'');
   ($FKong::cgi::form{'type'} || $type) =~ /^(\w+)$/ or FKong::Fatal("Internal Error: illegal char in field type.",'');
   my $new_type = $1;
   $FKong::cgi::keyword{'fieldId'} = $fieldId;
   $FKong::cgi::keyword{'tablesql'} = FKong::cgi::value_quote($tablesql);
   $FKong::cgi::keyword{'fieldName'} = FKong::cgi::value_quote($fieldName);
   $FKong::cgi::keyword{'type'} = FKong::cgi::value_quote($new_type);
   $FKong::cgi::keyword{'typeName'} = $typeNames{$new_type};
   $FKong::cgi::keyword{'typeName'} or FKong::Fatal("Internal Error: bad type.",'bad type for edit');
   $FKong::cgi::keyword{'min'} = defined $min ? FKong::cgi::value_quote($min) : "";
   $FKong::cgi::keyword{'max'} = defined $max ? FKong::cgi::value_quote($max) : "";
   $FKong::cgi::keyword{'width'} = defined $width ? FKong::cgi::value_quote($width) : "";
   $FKong::cgi::keyword{'height'} = defined $height ? FKong::cgi::value_quote($height) : "";
   if($new_type eq 'bool') {
      $FKong::cgi::keyword{'default'} = $default ? "CHECKED" : "";
   } else {
      $FKong::cgi::keyword{'default'} = defined $default ? FKong::cgi::value_quote($default) : "";
   };
   $FKong::cgi::keyword{'on_new_form'} = $on_new_form ? "CHECKED" : "";
   $FKong::cgi::keyword{'list_show'} = $list_show ? "CHECKED" : "";
   $FKong::cgi::keyword{'basic_search'} = $basic_search ? "CHECKED" : "";
   $FKong::cgi::keyword{'adv_search'} = $adv_search ? "CHECKED" : "";
   $FKong::cgi::keyword{'new_summary'} = $new_summary ? "CHECKED" : "";
   $FKong::cgi::keyword{'cmt_summary'} = $cmt_summary ? "CHECKED" : "";
   $FKong::cgi::keyword{'batch'} = $batch ? "CHECKED" : "";
   $FKong::cgi::keyword{'rset_on_dup'} = $rset_on_dup ? "CHECKED" : "";
   $FKong::cgi::keyword{'unique'} = $unique ? "CHECKED" : "";
   $FKong::cgi::keyword{'mod_by'} = defined $mod_by ? FKong::cgi::value_quote($mod_by) : "";
   $FKong::cgi::keyword{'modifyTS'} = Date::Format::time2str($FKong::config{'TimeFormat'},$modifyTS);
   $FKong::cgi::keyword{'create_by'} = defined $create_by ? FKong::cgi::value_quote($create_by) : "";
   $FKong::cgi::keyword{'createTS'} = Date::Format::time2str($FKong::config{'TimeFormat'},$createTS);
   $FKong::cgi::keyword{'sort'} = FKong::cgi::value_quote($sort);
   $FKong::cgi::keyword{'short_help'} = defined $short_help ? FKong::cgi::value_quote($short_help) : "";
   $FKong::cgi::keyword{'long_help'} = defined $long_help ? FKong::cgi::htmlQuoteForPre($long_help) : "";
   $FKong::cgi::keyword{'hidden'} = "<input type=hidden name=fieldId value=\"". FKong::cgi::value_quote($fieldId) ."\">\n".
                             "<input type=hidden name=type value=\"". FKong::cgi::value_quote($new_type) ."\">\n";
   $FKong::cgi::keyword{'formurl'} = FKong::cgi::Uri("list.html");
   # multi choice
   $FKong::cgi::keyword{'mult_choices'} = "";
   $dbh = FKong::db::SendSQL("DESCRIBE $tablesql $sqlName");
   my(undef,$enum) = $dbh->fetchrow_array();
   if($enum =~ /^enum\('(.*)'\)$/i) {  # if this field is an enum
      my @choices = split(/','/,$1);  
      foreach (@choices) { $_ = FKong::db::SqlUnquoteInside($_) };
      $FKong::cgi::keyword{'mult_choices'} = FKong::cgi::htmlQuoteForPre(join("\n",@choices));
   }
   my($has_index,$col2,$col3) = determine_current_index($sqlName,$tablesql);
   $FKong::cgi::keyword{'index'} = $has_index ? " CHECKED" : "";
   index_selects($fktable,$fieldId,$col2,$col3);
}

$FKong::table_func{'field/new.html'} = \&new_field;

# This will show the new field form. 
# This is called after the field type is decided.
sub new_field
{
   my($fktable) = @_;
   FKong::session::must_have_priv("edit_fields");
   $FKong::cgi::keyword{'formurl'} = FKong::cgi::Uri("list.html");
   my $type = $FKong::cgi::form{'type'};
   if(! defined $type || ! defined $typeNames{$type}) { 
      FKong::Fatal("Internal error: invalid or missing type",'bad type for new field');
   };
   $FKong::cgi::keyword{'typeName'} = $typeNames{$type};
   $FKong::cgi::keyword{'hidden'} = "<input type=hidden name=type value=\"". FKong::cgi::value_quote($type) ."\">\n";
   index_selects($fktable,-1);
   FKong::cgi::print_expanded_template("template/field_new-template.html"); 
}

# After a user has clicked the button to delete a field, this routine makes the change.
sub dirty_work_of_delete
{
   my($tablesql) = @_;
   FKong::session::must_have_priv("edit_fields");
   my $fieldId = $FKong::cgi::form{'fieldId'};
   if(!$fieldId) {  
      FKong::Fatal("Internal error: fieldId missing for delete",'');
   }
   my $dbh = FKong::db::SendSQL("SELECT sqlName FROM field_def WHERE fieldId = ". FKong::db::SqlQuote($fieldId));
   my $sqlName = $dbh->fetchrow_array();
   if(! defined $sqlName) {  # if already not there
      FKong::FKong::cgi::log_message("fieldId $fieldId already gone");
      return;
   }
   FKong::db::SendSQL("LOCK TABLES field_def WRITE, choice WRITE, $tablesql WRITE");
   FKong::db::ignore_cancel();
   FKong::db::SendSQL("DELETE FROM field_def WHERE fieldId = ". FKong::db::SqlQuote($fieldId));
   FKong::db::SendSQL("ALTER TABLE $tablesql DROP COLUMN $sqlName");
   FKong::db::SendSQL("UNLOCK TABLES");
}

my %minmax_types = ( integer => 1, hexint => 1, money => 1, mult_choice => 1, bool => 1);

# This is called after the user has committed to creating or chaning a field definition.
# We verify the form fields they sent describing the new table field.
# The return value is an SQL fragment to update the field_def table.
sub check_fields
{
   my($fktable,$fieldId) = @_;
   FKong::session::must_have_priv("edit_fields");
   if(defined $FKong::cgi::form{'fieldName'} && defined $FKong::cgi::form{'type'} && $FKong::cgi::form{'type'} eq 'constant' &&  # if type = constant
      length $FKong::cgi::form{'fieldName'} == 0 && defined $FKong::cgi::form{'default'}) {  # and no fieldName
      $FKong::cgi::form{'fieldName'} = $FKong::cgi::form{'default'};  # use the field default for the blank field name
   }
   foreach (qw/fieldName type/) {  # for mandatory non-blank fields
      FKong::Fatal("non-empty $_ required") if ! defined $FKong::cgi::form{$_} || $FKong::cgi::form{$_} eq "";
      FKong::cgi::trim($FKong::cgi::form{$_});
   }
   # check for conflicting fieldName
   my $dbh = FKong::db::SendSQL("SELECT fieldId FROM field_def\n".
                 "WHERE fieldName = ". FKong::db::SqlQuote($FKong::cgi::form{'fieldName'}) ."\n".
                 "AND fieldId != ". FKong::db::SqlQuote($fieldId) ."\n".
                 "AND tablesql = ". FKong::db::SqlQuote($$fktable{'tablesql'}) );
   if($dbh->fetchrow_array()) {
      FKong::Fatal("Error: That field name is already being used.  Please choose a different name.\n");
   };
   foreach (qw/min max short_help long_help width height/) {  # convert blank entries to undef so they map to NULL
      delete $FKong::cgi::form{$_} if defined $FKong::cgi::form{$_} && $FKong::cgi::form{$_} =~ /^\s*$/;  # we are bad people for modifying %form
   }
   $FKong::cgi::form{'mult_choices'} =~ s/^\s*\n//smg;  # remove blank lines
   my $type = $FKong::cgi::form{'type'};
   if(! $typeNames{$type}) {
      FKong::Fatal("Internal Error: Field type \"". FKong::cgi::htmlQuote($type) ."\" not understood.",'');
   };
   if($minmax_types{$type} && defined $FKong::cgi::form{'min'} &&  # if min referrs to textual length
      length $FKong::cgi::form{'min'} && $FKong::cgi::form{'min'} <= 0) {  # negative min makes no sense.  zero same as NULL.
      delete $FKong::cgi::form{'min'};
   }
   if($minmax_types{$type} && defined $FKong::cgi::form{'max'} &&   # if max refers to textual length
      length $FKong::cgi::form{'max'} && $FKong::cgi::form{'max'} <= 0) {  # negaive or zero max makes no sense
      delete $FKong::cgi::form{'max'};
   }
   #### FIXME we need to do more checking here.
   return "fieldName = ". FKong::db::SqlQuote($FKong::cgi::form{'fieldName'}) .",\n".
          "tablesql = ". FKong::db::SqlQuote($$fktable{'tablesql'}) .",\n".
          "type = ". FKong::db::SqlQuote($type) .",\n".
          "min = ". FKong::db::SqlQuote($FKong::cgi::form{'min'}) .",\n".
          "max = ". FKong::db::SqlQuote($FKong::cgi::form{'max'}) .",\n".
          "width = ". FKong::db::SqlQuote($FKong::cgi::form{'width'}) .",\n".
          "height = ". FKong::db::SqlQuote($FKong::cgi::form{'height'}) .",\n".
          "on_new_form = ". FKong::db::SqlQuote(! ! $FKong::cgi::form{'on_new_form'}) .",\n".
          "list_show = ". FKong::db::SqlQuote(! ! $FKong::cgi::form{'list_show'}) .",\n".
          "basic_search = ". FKong::db::SqlQuote(! ! $FKong::cgi::form{'basic_search'}) .",\n".
          "adv_search = ". FKong::db::SqlQuote(! ! $FKong::cgi::form{'adv_search'}) .",\n".
          "new_summary = ". FKong::db::SqlQuote(! ! $FKong::cgi::form{'new_summary'}) .",\n".
          "cmt_summary = ". FKong::db::SqlQuote(! ! $FKong::cgi::form{'cmt_summary'}) .",\n".
          "batch = ". FKong::db::SqlQuote(! ! $FKong::cgi::form{'batch'}) .",\n".
          "rset_on_dup = ". FKong::db::SqlQuote(! ! $FKong::cgi::form{'rset_on_dup'}) .",\n".
          "unique_ = ". FKong::db::SqlQuote(! ! $FKong::cgi::form{'unique'} && $type ne 'mult_choice' && $type ne 'bool') .",\n".
          "sort = ". FKong::db::SqlQuote($FKong::cgi::form{'sort'}) .",\n".
          "deflt = ". FKong::db::SqlQuote($FKong::cgi::form{'default'}) .",\n".
          "short_help = ". FKong::db::SqlQuote($FKong::cgi::form{'short_help'}) .",\n".
          "long_help = ". FKong::db::SqlQuote($FKong::cgi::form{'long_help'}) .",\n".
          "modifyTS = UNIX_TIMESTAMP(),\n".
          "mod_by = $FKong::cgi::keyword{'userid'}";
}

# Here we check the database schema to ensure the column name is unique.
# We try to make the column name resemble the fieldName.
sub choose_sql_name
{
   my($name,$tablesql) = @_;
   $name = "_$name";  # prefix all names with underscore
   $name =~ s/\s+/_/;  # convert spaces to underscores
   $name =~ tr/a-zA-Z_//cd;  # remove non alphabetics
   $name =~ tr/aeiou//d if length $name > 12;  # bastardize if long
   $name = substr($name,0,12);  # trim if longer than 12 chars
   $name = "f" if length $name == 0;
   my $dbh = FKong::db::SendSQL("DESCRIBE $tablesql $name");
   my $suffix = 1;
   while($dbh->fetchrow_array()) {
      $suffix++;
      $dbh = FKong::db::SendSQL("DESCRIBE $tablesql $name$suffix");
      FKong::Fatal("Internal Error: could not find unique sql name","uniq sql $name") if $suffix > 1000;
   }
   $name .= $suffix if $suffix != 1;
   return $name;
}

#  This routines assumes you already have the tables locked.
#  We redundantly store the choices of a multi_choice in the choice table, but the tablesql table enum is authoritative.
sub update_choice_table
{
   my($fieldId,$mult_choices) = @_;

   FKong::db::SendSQL("DELETE FROM choice WHERE fieldId = $fieldId");
   my @choices = map { chomp($_); FKong::db::SqlQuote($_) } split(/^/,$mult_choices);
   FKong::db::SendSQL("INSERT INTO choice (choice,fieldId) VALUES\n(". 
               join(",$fieldId),\n(",@choices) .",$fieldId)");
}

# For a mult_choice field, this function finds the choices which are retiring but still have some records.
sub find_retiring_values
{
   my($fieldId,$oldSqlName,$tablesql,@choices) = @_;

   my %choicehash;
   foreach (@choices) { $choicehash{$_} = 1 };
   my @deletes;
   my $dbh = FKong::db::SendSQL("SELECT $oldSqlName, COUNT(*) FROM $tablesql GROUP BY $oldSqlName");
   while(my($value,$count) = $dbh->fetchrow_array()) {
      next if ! defined $value;
      next if $choicehash{$value};  
      push(@deletes,$value) if defined $value && $value ne "";
   }
   if(scalar(@deletes)) {
      FKong::cgi::redirect("delete_choices.html?fieldId=$fieldId&". join("&",map { "choice=". FKong::cgi::urlQuote($_) } @deletes));
   };
   return @deletes;
}

# If this type can be a leading type of a multi-column index.
# Basically these type can be sufficiently non-unique.
my %more_indexable = ( mult_choice => 1, integer => 1, hex_int => 1, bool => 1, user => 1 );

# We do not save the index configuration in the field_def table.  
# Instead we check the actual index status on the table itself.
# This function does that investigative work.
# It is fairly MySQL specific.
sub determine_requested_index
{
   my($type,$tablesql) = @_;
   return ("","","","","") if ! $FKong::cgi::form{'index'} && ! $FKong::cgi::form{'unique'};  # unique requires an index
   return (1,"","","","") if ! $more_indexable{$type}; # if we don't support mult columns for this type
   my $should2 = $FKong::cgi::form{'index2'};
   $should2 =~ tr/a-zA-Z_//dc;  # remove bad characters whould should not be present
   return (1,"","","","") if ! $should2;
   my $dbh = FKong::db::SendSQL("DESCRIBE $tablesql ". FKong::db::SqlQuote($should2));
   my(undef,$def) = $dbh->fetchrow_array();  # if the requested second column does not exist
   return (1,"","","","") if ! $def;  # if the requested second column does not exist
   return (1,$should2,"(15)","","") if $def =~ /text/i;    # if second type is text
   my $should3 = $FKong::cgi::form{'index3'};
   $should3 =~ tr/a-zA-Z_//dc;  # remove bad characters whould should not be present
   return (1,$should2,"","","") if ! $should3;
   return (1,$should2,"","","") if $should2 eq $should3;
   $dbh = FKong::db::SendSQL("DESCRIBE $tablesql ". FKong::db::SqlQuote($should3));
   my(undef,$def3) = $dbh->fetchrow_array();  # if the requested third column does not exist
   return (1,$should2,"","","") if ! $def;  # if the requested third column does not exist
   return (1,$should2,"",$should3,"(15)") if $def3 =~ /text/i;    # if third type is text
   return (1,$should2,"",$should3,"");
}

# This function is used by both create and update. 
# We want to handle gracefully previously incorrect schema so the update case checks for an old index anyway.
# The tables should already be locked.  
sub fix_the_index
{
   my($fieldId,$type,$sqlName,$tablesql) = @_;
   my($found_key_name,$was_col2,$was_col3) = determine_current_index($sqlName,$tablesql);
   #print STDERR "found_key_name=$found_key_name was_col2=$was_col2 was_col3=$was_col3\n";
   my($req_index,$req_col2,$trim2,$req_col3,$trim3) =  determine_requested_index($type,$tablesql);
   #print STDERR "req_index=$req_index req_col2=$req_col2 req_col3=$req_col3\n";
   return if ! $found_key_name == ! $req_index && $was_col2 eq $req_col2 && $was_col3 eq $req_col3;  # if no change required
   FKong::db::SendSQL("ALTER TABLE $tablesql DROP INDEX $found_key_name") if $found_key_name;  # always drop to modify
   return if ! $req_index;  # if no index requested, then we are done
   if($more_indexable{$type}) {
      if($req_col3) {
         FKong::db::SendSQL("ALTER TABLE $tablesql ADD INDEX $sqlName ($sqlName,$req_col2,$req_col3$trim3)");
      } elsif($req_col2) {
         FKong::db::SendSQL("ALTER TABLE $tablesql ADD INDEX $sqlName ($sqlName,$req_col2$trim2)");
      } else {
         FKong::db::SendSQL("ALTER TABLE $tablesql ADD INDEX $sqlName ($sqlName)");
      }
   } elsif($type eq 'float') {
      FKong::db::SendSQL("ALTER TABLE $tablesql ADD INDEX $sqlName ($sqlName)");
   } elsif($type eq 'oneline') {
      FKong::db::SendSQL("ALTER TABLE $tablesql ADD INDEX $sqlName ($sqlName(15))");
   };
}

my %sql_type = ( integer => "int", hexint => 'int', float => "float", oneline => "text", money => 'int',
                 multiline => "text", bool => "bool", unix_ts => 'int unsigned', user => "mediumint unsigned" );

# This is called after the user clicks Update on the field update form.
# It is not called to create a field.
# Because the update form is a POST, this will redirect, and you should not show a page after calling this.
sub update
{
   my($fktable) = @_;
   my $fieldId = $FKong::cgi::form{'fieldId'};
   my $tablesql = $$fktable{'tablesql'};
   if(!$fieldId) {  # if they used the create-new form
      FKong::Fatal("Inernal error: fieldId not sent by browser.  Is the template missing {{hidden}} ?\n",'');
   }
   if($fieldId !~ /^\d+$/) {  # syntax error
      FKong::Fatal("browser error: fieldId must be numeric.\n",'');
   }
   # Check that the field ID exists.
   my $dbh = FKong::db::SendSQL("SELECT fieldId FROM field_def WHERE fieldId = $fieldId");
   if(! $dbh->fetchrow_array()) {
      FKong::Fatal("Error: The fieldId not found in database.  Was it deleted a moment ago by someone else?\n");
   };
   my $sql = check_fields($fktable,$fieldId);
   my $newType = $FKong::cgi::form{'type'};
   FKong::db::SendSQL("LOCK TABLES field_def WRITE, $tablesql WRITE, choice WRITE");
   FKong::db::ignore_cancel();
   $dbh = FKong::db::SendSQL("SELECT type, sqlName FROM field_def WHERE fieldId = $fieldId");
   my($oldType,$oldSqlName)= $dbh->fetchrow_array();
   my $sqlName = choose_sql_name($FKong::cgi::form{'fieldName'},$tablesql);
   (my $prefix = $sqlName) =~ s/\d*$//;  # remove trailing digits
   (my $oldPrefix = $oldSqlName || "") =~ s/\d*$//;  # remove trailing digits
   $sqlName = $oldSqlName if($prefix eq $oldPrefix);  # if same base prefix
   $dbh = FKong::db::SendSQL("DESCRIBE $tablesql $oldSqlName");
   my(undef,$old_def) = $dbh->fetchrow_array();
   my $new_def = $sql_type{$FKong::cgi::form{'type'}};
   my $verb = $old_def ? "CHANGE" : "ADD";
   $oldSqlName = '' if ! $old_def;   # so the SQL is correct
   if($new_def) {
      if($old_def ne $new_def || $sqlName ne $oldSqlName) {
         my($found_key_name) = determine_current_index($sqlName,$tablesql);
         FKong::db::SendSQL("ALTER TABLE $tablesql DROP INDEX $oldSqlName") if $found_key_name;
         FKong::db::SendSQL("ALTER TABLE $tablesql $verb COLUMN $oldSqlName $sqlName $new_def") 
      };
   } elsif($FKong::cgi::form{'type'} eq "mult_choice") {  # if field is created as multiple choice
      my $mult_choices = "";  # to update field_def field
      my @choices = split(/^/,$FKong::cgi::form{'mult_choices'});
      if($FKong::cgi::form{'mult_choices'} =~ /^\s*$/s) {  # if need to choose new choices automatically 
         $dbh = FKong::db::SendSQL("SELECT DISTINCT $oldSqlName FROM $tablesql GROUP BY $oldSqlName ORDER BY $oldSqlName");
         while(my $value = $dbh->fetchrow_array()) {  
            push(@choices,$value);
         };
      } 
      foreach (@choices) { 
         FKong::cgi::trim($_); 
      };
      if($oldType eq 'mult_choice') {  # else might need to retire some old choices
         push @choices, find_retiring_values($fieldId,$oldSqlName,$tablesql,@choices);   # this will redirect
      };
      foreach my $index (0..$#choices) { 
         delete $choices[$index] if $choices[$index] eq '';  # if line is empty
      };
      foreach (@choices) { 
         $mult_choices .= "$_\n"; 
         $_ = FKong::db::SqlQuote($_);  
      };
      FKong::db::SendSQL("ALTER TABLE $tablesql $verb COLUMN $oldSqlName $sqlName ENUM(\n  ". join(",\n  ",@choices) .")");
      FKong::db::SendSQL("UPDATE field_def SET mult_choices = ". FKong::db::SqlQuote($mult_choices) ." WHERE fieldId = $fieldId");
      update_choice_table($fieldId,$mult_choices);
   } elsif($old_def) {  # else must be changing to constant (go figure)
      FKong::db::SendSQL("ALTER TABLE $tablesql DROP COLUMN $sqlName");
   }
   FKong::db::SendSQL("UPDATE field_def SET sqlName = ". FKong::db::SqlQuote($sqlName) .",\n$sql WHERE fieldId = $fieldId");
   fix_the_index($fieldId,$FKong::cgi::form{'type'},$sqlName,$$fktable{'tablesql'});
   FKong::db::SendSQL("UNLOCK TABLES");
}

# This is called after the 'Create' button on the create form is clicked.
# Because that form is a POST, this will redirect to avoid double posts.  
# Thus, you should not print a page after calling called this function.
sub create
{
   my($fktable) = @_;
   my $tablesql = $$fktable{'tablesql'};
   my $sql = check_fields($fktable,-1);
   FKong::db::SendSQL("LOCK TABLES field_def WRITE, $tablesql WRITE, choice WRITE");
   my $sqlName = choose_sql_name($FKong::cgi::form{'fieldName'},$tablesql);
   my $mult_choices = "";
   my $type = $FKong::cgi::form{'type'} || FKong::Fatal("Internal Error: missing type.");
   if($sql_type{$type}) { 
      FKong::db::SendSQL("ALTER TABLE $tablesql ADD COLUMN $sqlName $sql_type{$type}");
   } elsif($type eq "mult_choice") {  # if field is created as multiple choice
      my @choices = split(/^/,$FKong::cgi::form{'mult_choices'});
      foreach (@choices) { 
         FKong::cgi::trim($_); 
         next if $_ eq "";  # line is empty
         $mult_choices .= "$_\n"; 
         $_ = FKong::db::SqlQuote($_) 
      };
      FKong::db::SendSQL("ALTER TABLE $tablesql ADD COLUMN $sqlName ENUM(\n  ". join(",\n  ",@choices) .")");
   }  # else must be constant 
   FKong::db::SendSQL("INSERT INTO field_def SET createTS = UNIX_TIMESTAMP(), create_by = $FKong::cgi::keyword{'userid'},\n".
               "$sql,\nsqlName = ". FKong::db::SqlQuote($sqlName) .",\nmult_choices = ". FKong::db::SqlQuote($mult_choices) );
   my $fieldId = FKong::db::last_insert_id();
   update_choice_table($fieldId,$mult_choices) if $type eq 'mult_choice';
   fix_the_index($fieldId,$type,$sqlName,$$fktable{'tablesql'});
   FKong::db::SendSQL("UNLOCK TABLES");
}

$FKong::table_func{'field/delete_choices.html'} = \&delete_choices;

# For multiple choice fields, when a choice goes away, this form lets them migrate the records which had the old choice.
sub delete_choices
{
   my($fktable) = @_;
   my $fieldId = $FKong::cgi::form{'fieldId'};
   FKong::Fatal('Internal Error: bad fieldId') if ! $fieldId || $fieldId !~ /^\d+$/;
   $FKong::cgi::keyword{'hidden'} = "<input type=hidden name=fieldId value=\"". FKong::cgi::value_quote($fieldId) ."\">\n";
   my $dbh = FKong::db::SendSQL("SELECT sqlName, type FROM field_def WHERE fieldId = $fieldId");
   my($sqlName,$type) = $dbh->fetchrow_array();
   FKong::Fatal("Internal Error: field doesn't exist") if ! $sqlName;
   FKong::Fatal("Internal Error: field not mult_choice") if $type ne 'mult_choice';
   $dbh = FKong::db::SendSQL("DESCRIBE $$fktable{'tablesql'} $sqlName");
   my(undef,$def) = $dbh->fetchrow_array();
   FKong::Fatal("Internal Error: table definition not enum") if $def !~ /^enum\('(.*)'\)$/i;
   my @old_choices = split(/','/,$1);
   foreach (@old_choices) { $_ = FKong::db::SqlUnquoteInside($_) };
   FKong::Fatal("Internal Error: no choice to delete") if ! $FKong::cgi::form{'choice'};
   my @deletes = split(/\0/,$FKong::cgi::form{'choice'});
   FKong::Fatal("Internal Error: no choices to delete") if 0 == scalar @deletes;
   my %deletehash;
   foreach (@deletes) { $deletehash{$_} = 1 };
   my @new_choices;
   foreach (@old_choices) {
      push(@new_choices,$_) if ! $deletehash{$_};
   }
   my @options = map { "  <option value=\"". FKong::cgi::value_quote($_) ."\">". FKong::cgi::htmlQuote($_) ."</option>\n"; } @new_choices;
   my $opt = "  <option value=\"** don't delete **\" SELECTED>** don't delete **</option>\n". join('',@options);
   my $form = '';
   foreach (@deletes) {
      $form .= "<tr><td>". FKong::cgi::htmlQuote($_) ."</td><td>--&gt;</td><td><select name=\"". FKong::cgi::value_quote("--$_") ."\">\n$opt</select></td><tr>\n"; 
   }
   $FKong::cgi::keyword{'form'} = $form;
   $FKong::cgi::keyword{'formurl'} = "list.html";
   FKong::cgi::print_expanded_template("template/delete_choice-template.html");
}

#  This function deletes choices of a multiple choice after mapping them to a new value.
#  It only shows stuff to the user if their is a problem.
sub migrate_choices
{
   my($tablesql,$pkey) = @_;
   my $fieldId = $FKong::cgi::form{'fieldId'};
   FKong::Fatal('Internal Error: bad fieldId') if ! $fieldId || $fieldId !~ /^\d+$/;
   my $dbh = FKong::db::SendSQL("SELECT sqlName, type FROM field_def WHERE fieldId = $fieldId");
   my($sqlName,$type) = $dbh->fetchrow_array();
   FKong::Fatal("Internal Error: field doesn't exist") if ! $sqlName;
   FKong::Fatal("Internal Error: field not mult_choice") if $type ne 'mult_choice';
   FKong::db::SendSQL("LOCK TABLES $tablesql WRITE, ${tablesql}_change WRITE");
   FKong::db::ignore_cancel();
   $dbh = FKong::db::SendSQL("DESCRIBE $tablesql $sqlName");
   my(undef,$def) = $dbh->fetchrow_array();
   FKong::Fatal("Internal Error: table definition not enum") if $def !~ /^enum\('(.*)'\)$/i;
   my @choices = split(/','/,$1);
   foreach (@choices) { $_ = FKong::db::SqlUnquoteInside($_) };
   my %valid_choices;
   map { $valid_choices{$_} = 1 } @choices;
   my $problems;
   my @keepers;
   foreach my $old_choice (@choices) {
      my $new_choice = $FKong::cgi::form{"--$old_choice"};
      if(! defined $new_choice || $new_choice eq "** don't delete **") {
         push(@keepers,$old_choice);
         next;
      }
      if(! $valid_choices{$new_choice}) {
         $problems .= "Internal Error: new choice ". FKong::cgi::htmlQuote($new_choice) ." for ". FKong::cgi::htmlQuote() ."not valid<br>\n";
      } else {
         FKong::db::SendSQL("INSERT INTO ${tablesql}_change ($pkey, fieldId, old_val, createTS, create_by) \n".
                     "SELECT $pkey, $fieldId, $sqlName, UNIX_TIMESTAMP(), $FKong::cgi::keyword{'userid'}\n".
                     "FROM $tablesql WHERE $sqlName = ". FKong::db::SqlQuote($old_choice) );
         FKong::db::SendSQL("UPDATE $tablesql SET $sqlName = ". FKong::db::SqlQuote($new_choice) ." WHERE $sqlName = ". FKong::db::SqlQuote($old_choice) );
      };
   }
   FKong::db::SendSQL("ALTER TABLE $tablesql MODIFY $sqlName ENUM(". join(',', map { FKong::db::SqlQuote($_) } @keepers) .")");
   FKong::Fatal($problems) if $problems;
   FKong::db::SendSQL("UNLOCK TABLES");
}

$FKong::table_func{'field/type.html'} = \&field_type;

# This shows the form which users can select the field type with.
# This form shows for new fields as well as modifications to existing fields.
sub field_type
{
   my($fktable) = @_;
   my $fieldId = $FKong::cgi::form{'fieldId'} || 0;
   my $dbh = FKong::db::SendSQL("SELECT fieldId, type FROM field_def WHERE fieldId = ". FKong::db::SqlQuote($fieldId));
   my($id,$type) = $dbh->fetchrow_array();
   $type = 'mult_choice' if ! $type;  # default value  (for new fields)
   foreach (keys %typeNames) {
      $FKong::cgi::keyword{"$_-title"} = $typeNames{$_};
      $FKong::cgi::keyword{$_} = ($_ eq $type) ? 'CHECKED' : '';
   };
   $FKong::cgi::keyword{'formurl'} = $fieldId ? "edit.html" : "new.html";
   $FKong::cgi::keyword{'hidden'} = "<input type=hidden name=fieldId value=\"". FKong::cgi::value_quote($fieldId) ."\">";
   FKong::cgi::print_expanded_template("template/field_type-template.html");
}

$FKong::table_func{'field/columns.html'} = \&column_choice_table;

# This prints the form that lets users indicate which fields they want to see, 
# when viewing the table of all fields.
sub column_choice_table
{
   my($fktable) = @_;  # ignore
   my $tablesql = 'field_def';
   $FKong::cgi::keyword{'formurl'} = "list.html";
   $FKong::cgi::keyword{'table'} = FKong::list::column_choice($tbl,$tablesql);
   FKong::cgi::print_expanded_template("template/column_choice-template.html");
}

sub dump_row
{
   my($tablesql,$where) = @_;

   my $answer = '';
   my @cols;
   my $dbh = FKong::db::SendSQL("SHOW COLUMNS FROM $tablesql");
   while(my @row = $dbh->fetchrow_array) {
      push(@cols,$row[0]);  # keep the column names
   }
   my $head = "INSERT INTO $tablesql (". join(", ",@cols) .") VALUES\n";
   $head =~ s/(.{60,75}) +/$1\n/g;  # wrap text
   $answer .= FKong::cgi::htmlQuoteForPre($head);
   $dbh = FKong::db::SendSQL("SELECT * FROM $tablesql $where");
   while(my @row = $dbh->fetchrow_array) {
      $answer .= FKong::cgi::htmlQuoteForPre("(". join(", ",map { FKong::db::SqlQuote($_) } @row) .")\n");  
   }
   return $answer;
}

# This does not drop the table, but merely return the SQL that would do it.
sub drop_table_sql
{
   my($fktable) = @_;  
   my $tablesql = $$fktable{'tablesql'}; 
   my $sql = "<pre>\n";
   $sql .= FKong::cgi::htmlQuoteForPre("DROP TABLE $tablesql;\n");
   $sql .= FKong::cgi::htmlQuoteForPre("DROP TABLE ${tablesql}_change;\n") if $$fktable{'changes'};
   $sql .= FKong::cgi::htmlQuoteForPre("DROP TABLE ${tablesql}_comment;\n") if $$fktable{'comments'};
   $sql .= FKong::cgi::htmlQuoteForPre("DELETE FROM field_def WHERE tablesql = ". FKong::db::SqlQuote($tablesql) .";\n");
   $sql .= FKong::cgi::htmlQuoteForPre("DELETE FROM fktable WHERE tablesql = ". FKong::db::SqlQuote($tablesql) .";\n");
   $sql .= "</pre>\n";
   return $sql;
}

$FKong::table_func{'show_create.html'} = \&show_create_table;

sub show_create_table
{
   my($fktable) = @_;  
   my $tablesql = $$fktable{'tablesql'}; 
   FKong::session::must_have_priv("view_fields");
   my $stuff = drop_table_sql($fktable);
   $stuff .= "<pre>\n";
   if($tablesql) {
      my $dbh = FKong::db::SendSQL("SHOW CREATE TABLE $tablesql");
      my(undef,$sql) = $dbh->fetchrow_array();
      $stuff .= FKong::cgi::htmlQuoteForPre($sql);
   };
   if($$fktable{'changes'}) {
      my $dbh = FKong::db::SendSQL("SHOW CREATE TABLE ${tablesql}_change");
      my(undef,$sql) = $dbh->fetchrow_array();
      $stuff .= FKong::cgi::htmlQuoteForPre("\n\n$sql");
   };
   if($$fktable{'comments'}) {
      my $dbh = FKong::db::SendSQL("SHOW CREATE TABLE ${tablesql}_comment");
      my(undef,$sql) = $dbh->fetchrow_array();
      $stuff .= FKong::cgi::htmlQuoteForPre("\n\n$sql");
   }; 
   $stuff .= "\n\n". dump_row('field_def',"WHERE tablesql = ". FKong::db::SqlQuote($tablesql)).
             "\n". dump_row('fktable',"WHERE tablesql = ". FKong::db::SqlQuote($tablesql)).
             "</pre>\n";
   $FKong::cgi::keyword{'stuff'} .= $stuff;
   FKong::cgi::print_expanded_template("template/show_create.html");
}

1;  # return code

