#!/usr/bin/perl -w
# -*- Mode: perl; indent-tabs-mode: nil -*-
#
# The contents of this file are subject to the Mozilla Public
# License Version 1.1 (the "License"); you may not use this file
# except in compliance with the License. You may obtain a copy of
# the License at http://www.mozilla.org/MPL/
#
# Software distributed under the License is distributed on an "AS
# IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
# implied. See the License for the specific language governing
# rights and limitations under the License.
#
# The Original Code is mozilla.org code.
#
# The Initial Developer of the Original Code is Holger
# Schurig. Portions created by Holger Schurig are
# Copyright (C) 1999 Holger Schurig. All
# Rights Reserved.
#
# Contributor(s): Holger Schurig <holgerschurig@nikocity.de>
#                 Terry Weissman <terry@mozilla.org>
#                 Dan Mosedale <dmose@mozilla.org>
#                 Dave Miller <justdave@syndicomm.com>
#                 Frederick Dean <software@fdd.com>
#
#
#
#
# Hey, what's this?
#
# 'checksetup.pl' is a script that is supposed to run during installation
# time and also after every upgrade.
#
# The goal of this script is to make the installation even more easy.
# It does so by doing things for you as well as testing for problems
# early.
#
# And you can re-run it whenever you want. Especially after DCVS
# gets updated you SHOULD rerun it. Because then it may update your
# SQL table definitions so that they are again in sync with the code.
#
# So, currently this module does:
#
#     - check for required perl modules
#     - set defaults for local configuration variables
#     - create and populate the data directory after installation
#     - set the proper rights for the *.cgi, *.html ... etc files
#     - check if the code can access MySQL
#     - creates the database 'bugs' if the database does not exist
#     - creates the tables inside the database if they don't exist
#     - automatically changes the table definitions of older BugZilla
#       installations
#     - populates the groups
#     - put the first user into all groups so that the system can
#       be administrated
#     - changes already existing SQL tables if you change your local
#       settings, e.g. when you add a new platform
#
# People that install this module locally are not supposed to modify
# this script. 
#
# Developers however have to modify this file at various places. To
# make this easier, I have added some special comments that one can
# search for.
#
#     To                                               Search for
#
#     add/delete local configuration variables         --LOCAL--
#     check for more prerequired modules               --MODULES--
#     change the defaults for local configuration vars --LOCAL--
#     update the assigned file permissions             --CHMOD--
#     add more MySQL-related checks                    --MYSQL--
#     change table definitions                         --TABLE--
#     add more groups                                  --GROUPS--
#     create initial administrator account            --ADMIN--
#
# Note: sometimes those special comments occur more then once. For
# example, --LOCAL-- is at least 3 times in this code!  --TABLE--
# also is used more than once. So search for every occurence!
#


use diagnostics;
use strict;

###########################################################################
# Check required module
###########################################################################

#
# Here we check for --MODULES--
#

print "\nChecking perl modules ...\n";
unless (eval "require 5.006") {
    die "Sorry, you need at least Perl 5.006\n";
}

# vers_cmp is adapted from Sort::Versions 1.3 1996/07/11 13:37:00 kjahds,
# which is not included with Perl by default, hence the need to copy it here.
# Seems silly to require it when this is the only place we need it...
sub vers_cmp {
  if (@_ < 2) { die "not enough parameters for vers_cmp" }
  if (@_ > 2) { die "too many parameters for vers_cmp" }
  my ($a, $b) = @_;
  my (@A) = ($a =~ /(\.|\d+|[^\.\d]+)/g);
  my (@B) = ($b =~ /(\.|\d+|[^\.\d]+)/g);
  my ($A,$B);
  while (@A and @B) {
    $A = shift @A;
    $B = shift @B;
    if ($A eq "." and $B eq ".") {
      next;
    } elsif ( $A eq "." ) {
      return -1;
    } elsif ( $B eq "." ) {
      return 1;
    } elsif ($A =~ /^\d+$/ and $B =~ /^\d+$/) {
      return $A <=> $B if $A <=> $B;
    } else {
      $A = uc $A;
      $B = uc $B;
      return $A cmp $B if $A cmp $B;
    }
  }
  @A <=> @B;
}

# This was originally clipped from the libnet Makefile.PL, adapted here to
# use the above vers_cmp routine for accurate version checking.
sub have_vers {
  my ($pkg, $wanted) = @_;
  my ($msg, $vnum, $vstr);
  no strict 'refs';
  printf("Checking for %15s %-9s ", $pkg, !$wanted?'(any)':"(v$wanted)");

  eval { my $p; ($p = $pkg . ".pm") =~ s!::!/!g; require $p; };

  $vnum = ${"${pkg}::VERSION"} || ${"${pkg}::Version"} || 0;
  $vnum = -1 if $@;

  if ($vnum < 0) {
    $vstr = "not found";
  }
  elsif ($vnum > 0) {
    $vstr = "found v$vnum";
  }
  else {
    $vstr = "found unknown version";
  }

  my $vok = (vers_cmp($vnum,$wanted) > -1);
  print ((($vok) ? "ok: " : " "), "$vstr\n");
  return $vok;
}

# Check versions of dependencies.  0 for version = any version acceptible

my @missing = ();
unless (have_vers("DBI","1.13"))          { push @missing,"DBI" }
unless (have_vers("DBD::mysql","1.2209")) { push @missing,"DBD::mysql" }
unless (have_vers("Date::Parse",0))       { push @missing,"Date::Parse" }
unless (have_vers("Date::Format",0))       { push @missing,"Date::Format" }
unless (have_vers("Digest::MD5",0))       { push @missing,"Digest::MD5" }

# If CGI::Carp was loaded successfully for version checking, it changes the
# die and warn handlers, we don't want them changed, so we need to stash the
# original ones and set them back afterwards -- justdave@syndicomm.com
my $saved_die_handler = $::SIG{__DIE__};
my $saved_warn_handler = $::SIG{__WARN__};
unless (have_vers("CGI::Carp",0))    { push @missing,"CGI::Carp" }
$::SIG{__DIE__} = $saved_die_handler;
$::SIG{__WARN__} = $saved_warn_handler;

if (@missing > 0) {
    print "\n\n";
    print "You are missing some Perl modules which are required by featurekong.\n";
    print "They can be installed by running (as root) the following:\n";
    foreach my $module (@missing) {
        print "   perl -MCPAN -e 'install \"$module\"'\n";
    }
    print "\n";
    print "\nAll of these packages are probably supplied with your distribution";
    print "\nmedia in precompiled form.  If so, I strongly recommend you use";
    print "\nthat instead of compiling source code from CPAN.";
    print "\nAlso, consider installing the security updated versions from";
    print "\nyour distribution when available.";
    print "\n";
    exit;
}

###########################################################################
# Check and update local configuration
###########################################################################

require "./featurekong_startup.pl";

my $my_db_host = $FKong::db_host;
   $my_db_host = $FKong::db_host;  # silence warnings
my $my_db_port = $FKong::db_port;
   $my_db_port = $FKong::db_port;  # silence warnings
my $my_db_name = $FKong::db_name;
   $my_db_name = $FKong::db_name;  # silence warnings
my $my_db_user = $FKong::db_user;
   $my_db_user = $FKong::db_user;  # silence warnings
my $my_db_pass = $FKong::db_pass;
   $my_db_pass = $FKong::db_pass;  # silence warnings


###########################################################################
# File permissions
###########################################################################

sub fixOwnership {
   my($filename,$username) = (@_);

   my $uid = $username;
   if($uid !~ /^\d+$/) {  # if username is not all numeric
      $uid   = getpwnam($username);
      if(! defined $uid) {
         die("Operating system's (not database's) user $username does not exist.");
      };
   };
   my $group = (stat($filename))[5];  # get group so we can leave it unchanged
   if(! defined $group) {
      print(STDERR "#### The file \"$filename\" does not exist!");
      return;
   };
   chown($uid,$group,$filename);  # okay if fails
   my $file_uid = (stat($filename))[4];   
   if($uid != $file_uid) {
      print(STDERR "The file $filename is not owned by $username ($uid != $file_uid)\n".
                   "and I can't fix it, but you can with...\n".
                   "\tsu -c \"chown $username $filename\"\n".
                   "...or rerunning checksetup.pl as root.\n");
      exit -1;
   }
}

sub fixMode {
   my($mode,@filenames) = (@_);

   foreach my $filename (@filenames) {
      chmod $mode, $filename;
      my $actualMode = (stat($filename))[2] & 07777;
      if($actualMode != $mode) {
         printf(STDERR "The file $filename has the wrong permissions, and \n".
                       "I can't fix it, but you can with...\n".
                       "\tsu -c \"chmod %o $filename\"\n".
                       "...or rerunning checksetup.pl as root.\n",$mode);
         exit -1;
      }
   };
}

###########################################################################
# Check MySQL setup
###########################################################################

my $bigHelp = <<"EOF"
This might have several reasons:

* MySQL is not running.
* MySQL is running, but the rights are incorrect. 
* You goofed the parameters in featurekong_startup.pl.
* There is an subtle problem with Perl, DBI, DBD::mysql and MySQL. 

If you have not already, the rights can be set up with a statement like...

mysql -p --user=root --host=localhost -e "GRANT SELECT,INSERT,UPDATE,DELETE,
INDEX,ALTER,CREATE,DROP,REFERENCES ON kong.* TO kong\@127.0.0.1 IDENTIFIED BY
'kong_db_password'; FLUSH PRIVILEGES;"

... but choose a better password.  The passord is asks for is not the
one being set, but the root of the database (different than root of unix).
This too must not be left blank.

EOF
;

#
# Check if we have access to --MYSQL--
#

# This settings are not yet changeable, because other code depends on
# the fact that we use MySQL and not, say, PostgreSQL.

my $db_base = 'mysql';

# No need to "use" this here.  It should already be loaded from the
# version-checking routines above, and this file won't even compile if
# DBI isn't installed so the user gets nasty errors instead of our
# pretty one saying they need to install it. -- justdave@syndicomm.com
#use DBI;

# get a handle to the low-level DBD driver
my $drh = DBI->install_driver($db_base)
    or die "Can't connect to the $db_base. Is the database installed and up and running?\n";

if (1) {
    # Do we have the database itself?

    my $sql_want = "3.22.5";  # minimum version of MySQL

# original DSN line was:
#    my $dsn = "DBI:$db_base:$my_db_name;$my_db_host;$my_db_port";
# removed the $db_name because we don't know it exists yet, and this will fail
# if we request it here and it doesn't. - justdave@syndicomm.com 2000/09/16
    my $dsn = "DBI:$db_base:;$my_db_host;$my_db_port";
    my $dbh = DBI->connect($dsn, $my_db_user, $my_db_pass)
      or die "\nI can't connect to the $db_base database. $bigHelp";
    printf("Checking for %15s %-9s ", "MySQL Server", "(v$sql_want)");
    my $qh = $dbh->prepare("SELECT VERSION()");
    $qh->execute;
    my ($sql_vers) = $qh->fetchrow_array;
    $qh->finish;

    # Check what version of MySQL is installed and let the user know
    # if the version is too old to be used with DCVS.
    if ( vers_cmp($sql_vers,$sql_want) > -1 ) {
        print "ok: found v$sql_vers\n\n";
    } else {
        die "Your MySQL server v$sql_vers is too old./n" . 
            "   DCVS requires version $sql_want or later of MySQL.\n" . 
            "   Please visit http://www.mysql.com/ and download a newer version.\n";
    }

    my @databases = $dbh->func('_ListDBs');
    unless (grep /^$my_db_name$/, @databases) {
       print "Creating database $my_db_name ...\n";
       $drh->func('createdb', $my_db_name, "$my_db_host:$my_db_port", $my_db_user, $my_db_pass, 'admin')
            or die " The '$my_db_name' database is not accessible. $bigHelp";
    }
    $dbh->disconnect if $dbh;
}

# now get a handle to the database:
my $connectstring = "dbi:$db_base:$my_db_name:host=$my_db_host:port=$my_db_port";
my $dbh = DBI->connect($connectstring, $my_db_user, $my_db_pass)
    or die "Can't connect to the table '$connectstring'.\n",
           "Have you read the DCVS Guide in the doc directory?  Have you read the doc of '$db_base'?\n";

END { $dbh->disconnect if $dbh }





###########################################################################
# Table definitions
###########################################################################

#
# The following hash stores all --TABLE-- definitions. This will be used
# to automatically create those tables that don't exist. 
#
# If you want to intentionally recreate the table, you can drop a table and re-run
# checksetup, e.g. like this:
#
#    $ mysql kong
#    mysql> drop table votes;
#    mysql> exit;
#    $ ./checksetup.pl
#
# If you change one of those field definitions, then also go below to the
# next occurence of the string --TABLE-- (near the end of this file) to
# add the code that updates older installations automatically.
#

# If we are creating the fktable table, then it is a new install so
# we will also create the feature table.
my $feature_table_flag = ! grep { $_ =~ /\.fktable\W*$/ } $dbh->tables;

my %table;
 
$table{'config'} = q{
    seq_num int unsigned not null auto_increment primary key,
    name varchar(32) not null,
    value mediumtext,
    mtime int unsigned,

    index(seq_num)  
};

# Here we define all the tables
$table{'fktable'} = q{
    fktableId int unsigned not null auto_increment primary key,
    name varchar(64) not null,
    recprefix varchar(32) not null,
    url varchar(64),
    tablesql varchar(64) not null, # table name, possibly with database too
    pkey varchar(64),  # column name of primary key 
    internal bool not null default 0,   # does FeatureKong require this one
    changes bool not null default 0,   # does it have a *_changes table
    comments bool not null default 0,   # does it have a *_comments table
    ext_mod bool not null default 0,  # can we not trust *_changes table to be complete
    down_maint mediumtext not null,   # down for maintenance
    advertise bool not null default 1,   # show in header and elsewhere
    create_by int unsigned,  # user.userid
    createTS int unsigned not null,
    mod_by int unsigned,     # user.userid
    modifyTS int unsigned not null,
    sort mediumint unsigned not null default 1000,

    unique(name),
    index(url),
    index(modifyTS),
    index(recprefix)
};


# The userid is nonzero if and only if they are logged in
# Remember, stealing a cookie is like stealing a username & password.
$table{'session'} = q{
    sessionId int unsigned not null auto_increment primary key,
    cookie varchar(32) not null,
    userid mediumint unsigned not null,
    lastused int unsigned not null,
    remoteip int unsigned not null,
    isvalid mediumint unsigned not null default 0,
    debug bool not null,

    unique(cookie),
    index(userid),
    index(lastused)  
};

$table{'more_session'} = q{
    cookie varchar(32) not null,
    name varchar(32) not null,
    value mediumblob not null,
    del_time int unsigned,

    unique(cookie,name)
};

# lastTS is the last login time.
# time is stored as unix timestamps (seconds since jan 1, 1970) instead of native time format
$table{'user'} = q{
    userid mediumint unsigned not null auto_increment primary key,
    username varchar(100) not null,
    email varchar(100) not null,  # just the stuff inside the angle brackets
    cryptpassword varchar(67),
    realname varchar(255),
    disabledtext mediumtext not null,
    privs int unsigned not null,  # bit field
    phone varchar(100),
    lastTS int unsigned not null,
    comment text not null,
    create_by int unsigned,  # user.userid
    createTS int unsigned not null,
    mod_by int unsigned,     # user.userid
    modifyTS int unsigned not null,
    email_pref mediumtext not null, # comma separated list of fieldId who do not qualify for email

    index(userid),
    index(modifyTS),
    unique(username),
    index(realname(10))
};


$table{'feature'} = q{
    featureId int unsigned not null auto_increment primary key,
    create_by int unsigned,  # user.userid
    createTS int unsigned not null,
    mod_by int unsigned,     # user.userid
    modifyTS int unsigned not null,

    # column below here are documented in field_def and are thus changeable
    _Title mediumtext,
    _Product enum('--','Widget'),
    _State enum('--','Proposed','Accepted','Scheduled','Development','Testing','Shipped'),
    _Priority int,
    _Dpndncs text,
    _Owner mediumint unsigned,
    _Dv_MnWks float,
    _Tst_MnWks float,
    _Definition text, 
    _Trgt_Rls enum('--','0.9','1.0','1.1','2.0'), 

    index(modifyTS),
    index(_Priority,_State),
    index(_Owner),
    index(_Product),
    index(_Title(15))
} if $feature_table_flag;


# Each feature has some properties.
# Some properties are multiple-choise where the legal choices are defined in this table.
$table{'field_def'} = q{
    fieldId mediumint unsigned not null auto_increment primary key,
    fieldName varchar(50) not null,  # shown to user
    tablesql varchar(50) not null, # This is table name.  must match fktable
    sqlName varchar(50) not null, # database column name.  User never sees
    type enum ('mult_choice', 'integer', 'oneline', 'multiline', 'bool', 'float', 'constant','user','unix_ts','hexint', 'money', 'ip') not null,
    mult_choices text,  # Non authoritative.  One per line.
    min int,  # only for integers
    max int,  # only for integers
    deflt text,   # default
    read_only bool not null,   # not editable by FeatureKong
    fixed_sql bool not null,   # type and sqlName cannot be changed
    on_edit_form bool not null default 1,     
    on_new_form bool not null default 1,   # is settable with by person creating feature
    list_show bool not null,     # for news page
    cmt_list_show bool not null,  # for news page
    basic_search bool not null,   # is it itemized on the search form
    adv_search bool not null default 1,
    new_summary bool not null default 0,  # does it appear on the new feature summary 
    cmt_summary bool not null default 0,  # does it appear on the new comment summary 
    batch bool not null default 0,  # can it be edited with the multiple feature edit form
    rset_on_dup bool not null default 0,  # Do we reset to default when duplicating a record
    unique_ bool not null default 0,  # note: constraint not enforce by db schema
    create_by int unsigned,  # user.userid
    createTS int unsigned not null,
    mod_by int unsigned,     # user.userid
    modifyTS int unsigned not null,
    sort mediumint unsigned not null default 1000,
    short_help mediumtext,
    long_help mediumtext,
    width mediumint unsigned default 80,  # of html form
    height mediumint unsigned default 8,  # of html form
    changeId int unsigned not null,  # for efficiency 

    index(fieldName),
    index(type,fieldName),
    index(tablesql,sort)
};

# This records changes of all field types for all field_def defined fields.  
$table{'feature_change'} = q{
    changeId int unsigned not null auto_increment primary key,
    featureId int unsigned not null,
    createTS int unsigned not null,
    create_by int unsigned not null,              # match with user.userid
    fieldId mediumint unsigned not null,
    old_val text,

    index(featureId,createTS),
    index(featureId,changeId),
    index(create_by,createTS),
    index(fieldId,createTS)
} if $feature_table_flag;

# This is essentially the error log
$table{'log'} = q{
    logId mediumint unsigned not null auto_increment primary key,
    createTS int unsigned not null,   # unix timestamp
    remoteip int unsigned not null,
    message mediumtext not null,
    pathfile mediumtext,
    create_by mediumint unsigned,  # match to user.userid
    mod_by int unsigned,     # user.userid
    modifyTS int unsigned not null,

    index(createTS),
    index(remoteip)
};

$table{'out_mail'} = q{
    outmailId mediumint unsigned not null auto_increment primary key,
    createTS int unsigned not null,   # unix timestamp
    create_by mediumint unsigned,  # match to user.userid
    mod_by int unsigned,     # user.userid
    modifyTS int unsigned not null,
    send_to mediumtext not null,
    subject mediumtext not null,
    message mediumtext not null,
    delivered bool not null default 0,  # was it given away to outbound SMTP server?

    index(delivered,createTS)
};


# This table is not authoritative and is only used for general text searches.
$table{'choice'} = q{
    choice varchar(50) not null,
    fieldId mediumint unsigned not null,

    unique(choice,fieldId),
    index(fieldId)
};

$table{'fktable_comment'} = q{
    commentId int unsigned not null auto_increment primary key,
    fktableId int unsigned not null,
    create_by int unsigned,  # user.userid
    createTS int unsigned not null,
    mod_by int unsigned,     # user.userid
    modifyTS int unsigned not null,
    format enum('text','html'),
    subject text,
    body text,
    #parentId int unsigned,  # match to commentId for threading
    changeId int unsigned not null, 

    index(fktableId,createTS),
    index(fktableId,changeId),
    index(create_by,createTS),
    fulltext(subject,body)
};

if($feature_table_flag) {
   $table{'feature_comment'} = $table{'fktable_comment'};
   $table{'feature_comment'} =~ s/fktableId/featureId/g;
};


=unused
$table{'attach'} = q{
    attachId mediumint unsigned not null auto_increment primary key,
    filesize int unsigned, 
    linktext mediumtext, 
    filename varchar(100), 
    attachTS int unsigned not null,
    comment mediumtext,
 
    index(attachId)
};
=cut



###########################################################################
# Create tables
###########################################################################

# Get a list of the existing tables (if any) in the database
my @tables = map { $_ =~ s/.*\.//; $_ } $dbh->tables;
#print 'Tables: ', join " ", @tables, "\n";

# go throught our %table hash and create missing tables
while (my ($tabname, $fielddef) = each %table) {
    next if grep { /^\W*$tabname\W*$/ } @tables;
    print "Creating table $tabname ...\n";

    $dbh->do("CREATE TABLE $tabname (\n$fielddef\n)")
        or die "Could not create table '$tabname'. Please check your '$db_base' access.\n";
}

###########################################################################
# Make sure fktable has internal listings
###########################################################################

sub ChangeFieldType ($$$);  # function prototype
sub AddField ($$$);  # function prototype
sub GetFieldDef ($$);  # function prototype

if(AddField('fktable','advertise','bool not null default 1')) {  # if field 'advertise' is added
   $dbh->do("UPDATE fktable SET advertise = ! internal");  # set default advertise bit
   $dbh->do("UPDATE fktable SET modifyTS = UNIX_TIMESTAMP() WHERE ! internal LIMIT 1");  # touch one so the header updates
}
AddField('fktable','down_maint','mediumtext not null');
AddField('fktable','sort','mediumint unsigned not null default 1000');

print "Populating fktable ...\n";

sub EnsureFktable
{
   my($name,$recprefix,$url,$tablesql,$pkey,$internal,$changes,$comments,$ext_mod) = @_;
   if(! fetch_one("SELECT tablesql FROM fktable WHERE tablesql = '$tablesql'")) {
      my $sth = $dbh->prepare("INSERT INTO fktable SET name = '$name', recprefix = '$recprefix',".
                              "url = '$url', tablesql = '$tablesql', internal = '$internal', ext_mod = '$ext_mod', ".
                              "changes = $changes, comments = $comments, pkey = '$pkey', advertise = ! '$internal', ".
                              "createTS = UNIX_TIMESTAMP(), modifyTS = UNIX_TIMESTAMP()");
      $sth->execute;
      return 1;
   };
   return 0;
}

#               name           recprefix    url    tablesql  pkey     internal changes comments ext_mod
EnsureFktable('FKTable',         'fkt',  'fktable','fktable','fktableId'  ,1,    0,      0,       0);
EnsureFktable('Field Def',   'field_def','field_def','field_def','fieldId',1,    0,      0,       0);
EnsureFktable('Error Log',      'log',   'log',    'log',    'logId'      ,1,    0,      0,       0);
EnsureFktable('Outbound Email', 'out_mail','out_mail','out_mail','outmailId',1,    0,      0,       0);
EnsureFktable('Feature',          'f',  'feature','feature','featureId'   ,0,    1,      1,       0) if $feature_table_flag;
#EnsureFktable('User',            'user', 'user2',  'user',   'userid'     ,1,    0,      0,       0);
$dbh->do("DELETE FROM fktable WHERE tablesql = 'user'");
EnsureFktable('Session',    'session',  'session','session','sessionId'      ,1,    0,      0,       0);

###########################################################################
# Make sure field_def has internal listings
###########################################################################

if(${GetFieldDef('field_def','type')}[1] !~ /'ip'/) {  # if field type of ip does not exist
   ChangeFieldType('field_def','type',"enum('mult_choice','integer','oneline','multiline','bool','float','constant','user','unix_ts','hexint','money','ip') not null");
}

print "Populating field_def ...\n";

sub EnsureFieldDef
{
   my($read_only,$fieldName,$tablesql,$sqlName,$type,$basic_search,$sort,$short_help) = @_;
   my $internal = fetch_one("SELECT internal FROM fktable WHERE tablesql = '$tablesql'");
   my $fixed_sql = ($sqlName !~ /^_/);  # fixed_sql if sqlName does not begin with underscore
   if(! fetch_one("SELECT tablesql FROM field_def WHERE fieldName = '$fieldName' AND tablesql = '$tablesql'")) {
      my $list_show = $type ne 'multiline';
      my $sth = $dbh->prepare("INSERT INTO field_def SET fieldName = '$fieldName', tablesql = '$tablesql', ".
                              "sqlName = '$sqlName', type = '$type', basic_search = '$basic_search', sort = '$sort',".
                              "short_help = '$short_help', list_show = '$list_show', batch = 0,".
                              "fixed_sql = $fixed_sql,".
                              "createTS = UNIX_TIMESTAMP(), modifyTS = UNIX_TIMESTAMP()");
      $sth->execute;
      # handle the default
      if($type ne 'oneline' && $type ne 'multiline') {
         my $qh = $dbh->prepare("DESCRIBE $tablesql $sqlName");
         $qh->execute;
         my(undef,undef,undef,undef,$default) = $qh->fetchrow_array();
         $qh = $dbh->prepare("UPDATE field_def SET deflt = ? WHERE tablesql = '$tablesql' AND sqlName = '$sqlName'");
         $qh->execute($default);
      }
   };
   # force read_only but not not-read_only
   $dbh->do("UPDATE field_def SET read_only = 1 WHERE tablesql = '$tablesql' AND sqlName = '$sqlName'") if $read_only;
}

if(! GetFieldDef("field_def", "fixed_sql")) {  # if field_def has no fixed_sql parameter
   AddField ('field_def','fixed_sql','bool not null');
   $dbh->do("UPDATE field_def SET fixed_sql = 1 WHERE sqlName NOT LIKE '_%'");
   my $qh = $dbh->prepare("SELECT tablesql FROM fktable WHERE ! internal");
   $qh->execute;
   while(1) {  # for every table listed in fktable
      my($tablesql) = $qh->fetchrow_array;
      last if ! defined $tablesql;
      #              fieldName        tablesql  sqlName      type     basic_search sort short_help
      if(GetFieldDef($tablesql,"create_by")) {
         EnsureFieldDef(1,'Creator',         $tablesql,'create_by','user'   ,0,        10040, '');
      };
      if(GetFieldDef($tablesql,"createTS")) {
         EnsureFieldDef(1,'Create Time',     $tablesql,'createTS', 'unix_ts',0,        10042, '');
      };
      if(GetFieldDef($tablesql,"mod_by")) {
         EnsureFieldDef(1,'Last Modifier',   $tablesql,'mod_by',   'user'   ,0,        10044, '');
      };
      if(GetFieldDef($tablesql,"modifyTS")) {
         EnsureFieldDef(1,'Last Modify Time',$tablesql,'modifyTS', 'unix_ts',0,        10046, '');
      };
   };
   $dbh->do("UPDATE field_def SET on_new_form = 0 WHERE sqlName IN ('create_by','createTS','mod_by','modifyTS')");
};

#         read_only fieldName   tablesql  sqlName      type     basic_search sort short_help
EnsureFieldDef(0,'Name',         'fktable','name',      'oneline',1,           100, 'Human readable moniker.');
EnsureFieldDef(0,'Record Prefix','fktable','recprefix', 'oneline',1,           110, 'Append to beginning of record number.');
EnsureFieldDef(0,'Url',          'fktable','url',       'oneline',1,           120, 'Cannot contain slashes.');
EnsureFieldDef(1,'Table SQL',    'fktable','tablesql',  'oneline',1,           130, 'Can include database name.');
EnsureFieldDef(1,'Internal',     'fktable','internal',  'bool',   0,           140, 'FeatureKong needs this table design unaltered.');
EnsureFieldDef(1,'Changes',      'fktable','changes',   'bool',   0,           160, 'Does this have a _changs table?');
EnsureFieldDef(1,'Comments',     'fktable','comments',  'bool',   0,           170, 'Does this have a _comments table?');
EnsureFieldDef(0,'Ext Mod',      'fktable','ext_mod',   'bool',   0,           180, 'Does another aplication modify it without updating the _changes table?');
EnsureFieldDef(0,'Advertise',    'fktable','advertise', 'bool',   0,           200, 'Display this table in the header and elsewhere.');
EnsureFieldDef(0,'Sort',         'fktable','sort',      'integer',0,           210, 'Ordering in the header and elsewhere.');
EnsureFieldDef(0,'Down Maint',   'fktable','down_maint','multiline',0,         400, 'Down for maintenance message. Only table designers can access it.');

#EnsureFieldDef(1,'UserID',       'user',    'userid',   'integer',   0,         100, '');
#EnsureFieldDef(1,'Username',     'user',    'username', 'oneline',   0,         110, '');
#EnsureFieldDef(1,'Email',        'user',    'email',    'oneline',   0,         120, '');
#EnsureFieldDef(1,'RealName',     'user',    'realname', 'oneline',   0,         130, '');
#EnsureFieldDef(1,'DisabledText', 'user',   'disabledtext','multiline',0,        140, '');
#EnsureFieldDef(1,'Privs',        'user',    'privs',    'hexint',    0,         150, '');
#EnsureFieldDef(1,'Phone',        'user',    'phone',    'oneline',   0,         160, '');
#EnsureFieldDef(1,'LastTS',       'user',    'lastTS',   'unix_ts',   0,         170, '');
#EnsureFieldDef(1,'Comment',      'user',    'comment',  'multiline', 0,         180, '');
$dbh->do("DELETE FROM field_def WHERE tablesql = 'user'");

EnsureFieldDef(1,'Name',        'field_def','fieldName',  'oneline',   0,         110, '');
EnsureFieldDef(1,'Table SQL',   'field_def','tablesql',   'oneline',   1,         120, '');
EnsureFieldDef(1,'SQL Name',    'field_def','sqlName',    'oneline',   0,         130, '');
EnsureFieldDef(1,'Type',        'field_def','type',       'multi_choice',0,       140, '');
EnsureFieldDef(1,'Multi-Choices','field_def','mult_choices','multiline', 0,       150, '');
EnsureFieldDef(0,'Min',         'field_def','min',        'integer',   0,         160, '');
EnsureFieldDef(0,'Max',         'field_def','max',        'integer',   0,         170, '');
EnsureFieldDef(0,'Default',     'field_def','deflt',      'oneline',   0,         180, '');
EnsureFieldDef(1,'Read Only',   'field_def','read_only',  'bool',      0,         185, '');
EnsureFieldDef(1,'Fixed SQL',   'field_def','fixed_sql',  'bool',      0,         188, '');
EnsureFieldDef(0,'On New Form', 'field_def','on_new_form','bool',      0,         190, '');
EnsureFieldDef(0,'List Show',   'field_def','list_show',  'bool',      0,         200, '');
EnsureFieldDef(0,'Comment List Show','field_def','cmt_list_show','bool',0,        205, '');
EnsureFieldDef(0,'Basic Search','field_def','basic_search','bool',     0,         210, '');
EnsureFieldDef(0,'Adv Search',  'field_def','adv_search', 'bool',      0,         220, '');
#EnsureFieldDef(1,'New Summary','field_def','new_summary','bool',      0,         230, '');
#EnsureFieldDef(1,'Comment Summary','field_def','cmt_summary','bool',  0,         240, '');
EnsureFieldDef(1,'Batchable',   'field_def','batch',      'bool',      0,         250, '');
EnsureFieldDef(1,'Unique',      'field_def','unique_',    'bool',      0,         260, '');
EnsureFieldDef(0,'Sort',        'field_def','sort',       'integer',   0,         280, '');
EnsureFieldDef(0,'Short Help',  'field_def','short_help', 'oneline',   0,         290, '(HTML)');
EnsureFieldDef(0,'Long Help',   'field_def','long_help',  'multiline', 0,         300, '(HTML)');

EnsureFieldDef(1,'Remote IP',   'log','remoteip',   'ip',  0,         300, '');
EnsureFieldDef(1,'Message',     'log','message',    'multiline',0,         400, '');
EnsureFieldDef(1,'Path',        'log','pathfile',   'onelin',   0,         500, '');

EnsureFieldDef(1,'Mail ID', 'out_mail', 'outmailId',   'integer',  0,         100, '');
EnsureFieldDef(1,'Delivered',   'out_mail', 'delivered',   'bool',     0,         150, '');
EnsureFieldDef(1,'Send To',     'out_mail', 'send_to',     'oneline',  0,         200, '');
EnsureFieldDef(1,'Subject',     'out_mail', 'subject',     'oneline',  0,         300, '');
#EnsureFieldDef(1,'Message',     'out_mail', 'message',     'multiline',  0,         400, '');

#EnsureFieldDef(1,'Cookie',    'session',   'cookie',   'oneline',  0,         100, '');
EnsureFieldDef(1,'User',      'session',   'userid',   'user',     0,         200, '');
EnsureFieldDef(1,'Last Used', 'session', 'lastused',   'unix_ts',  0,         300, '');
EnsureFieldDef(1,'Remote IP', 'session', 'remoteip',   'ip',  0,         400, '');
EnsureFieldDef(1,'Is Valid',  'session',  'isvalid',   'bool',     0,         500, '');
EnsureFieldDef(0,'Debug',     'session',  'isvalid',   'bool',     0,         600, '');


###########################################################################
# Detect changed local settings
###########################################################################

sub GetFieldDef ($$)
{
    my ($table, $field) = @_;
    my $sth = $dbh->prepare("SHOW COLUMNS FROM $table");
    $sth->execute;

    while (my $ref = $sth->fetchrow_arrayref) {
        next if $$ref[0] ne $field;
        return $ref;
   }
}

sub GetIndexDef ($$)
{
    my ($table, $field) = @_;
    my $sth = $dbh->prepare("SHOW INDEX FROM $table");
    $sth->execute;

    while (my $ref = $sth->fetchrow_arrayref) {
        next if $$ref[2] ne $field;
        return $ref;
   }
}

sub CountIndexes ($)
{
    my ($table) = @_;
    
    my $sth = $dbh->prepare("SHOW INDEX FROM $table");
    $sth->execute;

    if ( $sth->rows == -1 ) {
      die ("Unexpected response while counting indexes in $table:" .
           " \$sth->rows == -1");
    }
    
    return ($sth->rows);
}

sub DropIndexes ($)
{
    my ($table) = @_;
    my %SEEN;

    # get the list of indexes
    #
    my $sth = $dbh->prepare("SHOW INDEX FROM $table");
    $sth->execute;

    # drop each index
    #
    while ( my $ref = $sth->fetchrow_arrayref) {
      
      # note that some indexes are described by multiple rows in the
      # index table, so we may have already dropped the index described
      # in the current row.
      # 
      next if exists $SEEN{$$ref[2]};

      my $dropSth = $dbh->prepare("ALTER TABLE $table DROP INDEX $$ref[2]");
      $dropSth->execute;
      $dropSth->finish;
      $SEEN{$$ref[2]} = 1;

    }

}
#
# Check if the enums in the bugs table return the same values that are defined
# in the various locally changeable variables. If this is true, then alter the
# table definition.
#

sub CheckEnumField ($$@)
{
    my ($table, $field, @against) = @_;

    my $ref = GetFieldDef($table, $field);
    #print "0: $$ref[0]   1: $$ref[1]   2: $$ref[2]   3: $$ref[3]  4: $$ref[4]\n";
    
    $_ = "enum('" . join("','", @against) . "')";
    if ($$ref[1] ne $_) {
        print "Updating field $field in table $table ...\n";
        $_ .= " NOT NULL" if $$ref[3];
        $dbh->do("ALTER TABLE $table
                  CHANGE $field
                  $field $_");
    }
}

###########################################################################
# Define some defaults (once at first installation time)
###########################################################################

sub fetch_one {
   my($sql) = @_;
   my $sth = $dbh->prepare($sql);
   $sth->execute;
   my @row = $sth->fetchrow_array;
   return @row if wantarray;
   return $row[0];
}

# if we need to create the feature table.
if($feature_table_flag && fetch_one("SELECT COUNT(*) FROM field_def WHERE tablesql = 'feature'") == 0) {
   print("Creating fields for feature table.\n");
   my $sth = $dbh->prepare("INSERT INTO field_def\n".
      "(fieldName       ,sqlName       ,type         ,min ,deflt,short_help,sort,basic_search) VALUES\n".
      "('Title'         ,'_Title'      ,'oneline'    ,NULL,''   ,''        ,100 ,1),\n".
      "('Product'       ,'_Product'    ,'mult_choice',NULL,'--' ,''        ,200 ,1),\n".
      "('State'         ,'_State'      ,'mult_choice',NULL,'--' ,''        ,200 ,1),\n".
      "('Priority'      ,'_Priority'   ,'integer'    ,NULL,'100','(1 is most urgent)',200,1),\n".
      "('Target Release','_Trgt_Rls'   ,'mult_choice',NULL,'1.0',''        ,200 ,1),\n".
      "('Owner'         ,'_Owner'      ,'user'       ,NULL,''   ,''        ,250 ,0),\n".
      "('Dependencies'  ,'_Dpndncs'    ,'oneline'    ,NULL,''   ,''        ,260 ,0),\n".
      "('Dev. Man-Weeks','_Dv_MnWks'   ,'float'      ,0   ,'100',''        ,300 ,0),\n".
      "('Test Man-Weeks','_Tst_MnWks'  ,'float'      ,0   ,'100',''        ,300 ,0),\n".
      "('Definition'    ,'_Definition' ,'multiline'  ,NULL,''   ,''        ,400 ,0)\n");
   $sth->execute;
   # fix fixed_sql
   $sth = $dbh->prepare("UPDATE field_def SET fixed_sql = 1 WHERE tablesql = 'feature' AND sqlName NOT LIKE '\_%'"); 
   # fix timestamps and userids
   $sth = $dbh->prepare("UPDATE field_def SET createTS = UNIX_TIMESTAMP(), modifyTS = UNIX_TIMESTAMP(),\n".  
                            "tablesql = 'feature' WHERE createTS = 0");
   $sth->execute;
   # check sql case and fix mult_choice stuff
   $sth = $dbh->prepare("DESCRIBE feature");
   $sth->execute;
   while(my($sqlName,$def) = $sth->fetchrow_array) {
      # check for exact case match
      my $sth2 = $dbh->prepare("SELECT sqlName FROM field_def WHERE sqlName = ?");
      $sth2->execute($sqlName);
      my($name) = $sth2->fetchrow_array();
      ! defined $name || $name eq $sqlName or die "field_def.sqlName mismatch with feature table for $sqlName. Case problem?";
      # fix the mult_choices field of field_def
      next unless $def =~ /^enum\('(.*)'\)$/;
      my @choices = split(/','/,$1);
      my $mult_choices = "";  # to update field_def field
      foreach (@choices) { s/\\(.)/$1/sg; $mult_choices .= "$_\n"; }  # unescape
      $sth2 = $dbh->prepare("UPDATE field_def SET mult_choices = ? WHERE sqlName = ?");
      $sth2->execute($mult_choices,$sqlName);
      # fix the choice table (used for searches)
      $sth2 = $dbh->prepare("SELECT fieldId FROM field_def WHERE sqlName = ?");
      $sth2->execute($sqlName);
      my $fieldId = $sth2->fetchrow_array();
      if($fieldId) {
         my $sth2 = $dbh->prepare("INSERT INTO choice SET choice = ?, fieldId = ?");
         foreach (@choices) { $sth2->execute($_,$fieldId) };
      };
   }
   $sth = $dbh->prepare("UPDATE field_def SET new_summary = 1 WHERE sqlName IN ('_Title','_Product','_Priority','_Definition')");
   $sth->execute();
   $sth = $dbh->prepare("UPDATE field_def SET cmt_summary = 1 WHERE sqlName IN ('_Title','_Product','_Priority','_State')");
   $sth->execute();
   $sth = $dbh->prepare("UPDATE field_def SET list_show = 1 WHERE type NOT IN ('multiline')");
   $sth->execute();
   $sth = $dbh->prepare("UPDATE field_def SET batch = 1 WHERE type IN ('mult_choice','integer','float','user','bool')");
   $sth->execute();
   $sth = $dbh->prepare("UPDATE field_def SET unique_ = 1 WHERE sqlName IN ('_Title')");
   $sth->execute();
   $sth = $dbh->prepare("UPDATE field_def SET cmt_list_show = 1 WHERE sqlName IN ('_Title')");
   $sth->execute();
};

###########################################################################
# Create Special Users 
###########################################################################

sub create_user_if_needed
{
   my($userid,$username,$realname,$privs,$disabledtext,$comment) = @_;

   if(! fetch_one("SELECT userid FROM user WHERE userid = $userid")) {
      print "Creating special user $username.\n";
      my $sth = $dbh->prepare("INSERT INTO user SET userid = ?, createTS = UNIX_TIMESTAMP(), modifyTS = UNIX_TIMESTAMP(),\n".  
                            "username = ?, realname = ?, privs = ?, disabledtext = ?, comment = ?, email = '',\n".
                            "cryptpassword = 'false', phone = ''");
      $sth->execute($userid,$username,$realname,$privs,$disabledtext,$comment);
   }
} 


create_user_if_needed(99,"Anonymous","Anonymous User",0x800,'',
"People who have not logged in, operate as this user.  (#99)");
create_user_if_needed(98,"Applicants","Applicant User",0x1800,'',
"New applicants receive privileges and disabled text just like\n".
"this user.  Nobody should log into this account. (#98)");
create_user_if_needed(97,"New","New User",0x1800,'',
"This user's privileges are the default for the new user form.\n".
"Nobody should log into this account. (#97)");

###########################################################################
# Create Administrator 
###########################################################################

sub bailout {   # this is just in case we get interrupted while getting passwd
    system("stty","echo"); # re-enable input echoing
    exit 1;
}
sub getPassword {
    $SIG{HUP}  = \&bailout;
    $SIG{INT}  = \&bailout;
    $SIG{QUIT} = \&bailout;
    $SIG{TERM} = \&bailout;
    system("stty","-echo");  # disable input echoing
    my $pass = <STDIN>;
    system("stty","echo"); # re-enable input echoing
    $SIG{HUP}  = 'DEFAULT'; # and remove our interrupt hooks
    $SIG{INT}  = 'DEFAULT';
    $SIG{QUIT} = 'DEFAULT';
    $SIG{TERM} = 'DEFAULT';
    print("\n");
    chomp($pass);  # remove trailing newline
    return $pass;
}

my $new_admin_privs = 65535;

sub adminList {
   my @admins = ();
   my $sth = $dbh->prepare("SELECT username, privs FROM user");
   $sth->execute;
   while( my($username,$privs) = $sth->fetchrow_array ) {
      push(@admins,$username) if (($privs & 3) == 3);
   };
   return @admins;
}

my @admins = adminList();
if($#admins > -1) {
   print("\nYou already have some administrators... ". join(" ",@admins) ."\n");
} else {  # else we don't have an administrator but need one.
   print("\nYou do not have an administrator account, so we need to set one up.\n".
         "If you make a mistake, use control-C and run checksetup.pl again.\n".
         "What username will the administrator use? ");
   my $username = <STDIN>;
   chomp($username);
   my $sth = $dbh->prepare("SELECT userid, realname FROM user WHERE username = ?");
   $sth->execute($username);
   if( my($userid,$realname) = $sth->fetchrow_array ) {
      print("\nUser $username ($realname) already exists.\nShould we promote her to administrator? ");
      my $confirm = <>;
      exec($0) unless $confirm =~ /^y/i;  # restart unless response begins with y
      my $sth = $dbh->prepare("UPDATE user SET privs = $new_admin_privs | 65535, disabledtext = '' WHERE userid = ?");
      $sth->execute($userid);
   } else {  # else the user does not exist
      print("\nWhat is ${username}'s real name? ");
      my $realname = <STDIN>;
      chomp($realname);
      print("\nWhat is ${username}'s email address? ");
      my $email = <STDIN>;
      chomp($email);
      my($pass) = ("a");
      while(1) {  # while we try to agree on a password
         print("\nWhat is ${username}'s password? ");
         $pass = getPassword();
         if(length($pass) < 8) {
            print("\nYour password was too short. Please try again with 8 or more chars.\n");
            next;
         }
         if($pass =~ /^[a-zA-Z0-9]+$/) {  # is all alphanumeric
            print("\nYour password was all alphanumeric. Please try again.\n");
            next;
         }
         print("again? ");
         my $pass2 = getPassword();
         if($pass ne $pass2) {
            print("\nThose passwords didn't match, please try again.\n");
            next;
         };
         last;   # stop this password madness
      };
      require FKong::user;  # can't be "use", we need this at run (not compile) time.
      FKong::user::set_user(undef,$username,$pass,$realname,$email,"",'created by checksetup.pl',$new_admin_privs);
      print("Okay, I have created an administrator account for $username\n\n");
   };
}


###########################################################################
# Update the tables to the current definition
###########################################################################

#
# As time passes, fields in tables get deleted, added, changed and so on.
# So we need some helper subroutines to make this possible:
#

sub ChangeFieldType ($$$)
{
    my ($table, $field, $newtype) = @_;

    my $ref = GetFieldDef($table, $field);
    #print "0: $$ref[0]   1: $$ref[1]   2: $$ref[2]   3: $$ref[3]  4: $$ref[4]\n";

    my $oldtype = $ref->[1];
    if (! $ref->[2]) {
        $oldtype .= qq{ not null};
    }
    if ($ref->[4]) {
        $oldtype .= qq{ default "$ref->[4]"};
    }

    if ($oldtype ne $newtype) {
        print "Updating field type \"$field\" in table \"$table\" ...\n";
        print "old: $oldtype\n";
        print "new: $newtype\n";
#        'not null' should be passed as part of the call to ChangeFieldType()
#        $newtype .= " NOT NULL" if $$ref[3];
        $dbh->do("ALTER TABLE $table
                  CHANGE $field
                  $field $newtype");
    }
}

sub RenameField ($$$)
{
    my ($table, $field, $newname) = @_;

    my $ref = GetFieldDef($table, $field);
    return unless $ref; # already fixed?
    #print "0: $$ref[0]   1: $$ref[1]   2: $$ref[2]   3: $$ref[3]  4: $$ref[4]\n";

    if ($$ref[1] ne $newname) {
        print "Updating field \"$field\" in table \"$table\" ...\n";
        my $type = $$ref[1];
        $type .= " NOT NULL" if $$ref[3];
        $dbh->do("ALTER TABLE $table
                  CHANGE $field
                  $newname $type");
    }
}

# returns true if field is added, false if it already existed
sub AddField ($$$)
{
    my ($table, $field, $definition) = @_;

    TableExists($table) or return;
    my $ref = GetFieldDef($table, $field);
    return 0 if $ref; # already added?

    print "Adding new field \"$field\" to table \"$table\" ...\n";
    $dbh->do("ALTER TABLE $table
              ADD COLUMN $field $definition");
    return 1;
}

sub DropField ($$)
{
    my ($table, $field) = @_;

    my $ref = GetFieldDef($table, $field);
    return unless $ref; # already dropped?

    print "Deleting unused field \"$field\" from table \"$table\" ...\n";
    $dbh->do("ALTER TABLE $table
              DROP COLUMN $field");
}

# this uses a mysql specific command. 
sub TableExists ($)
{
   my ($table) = @_;
   my @tables;
   my $dbtable;
   my $exists = 0;
   my $sth = $dbh->prepare("SHOW TABLES");
   $sth->execute;
   while ( ($dbtable) = $sth->fetchrow_array ) {
      if ($dbtable eq $table) {
         $exists = 1;
      } 
   } 
   return $exists;
}   


AddField ('field_def','width','mediumint unsigned default 80');
AddField ('field_def','height','mediumint unsigned default 8');

AddField ('field_def','cmt_list_show','bool not null');
AddField ('field_def','read_only','bool not null');

# recreate the config table using seq_num instead of mtime as primary key 
if(! GetFieldDef('config', 'seq_num')) {
    my @configs;

    my $qh = $dbh->prepare("SELECT name, value FROM config");
    $qh->execute;
    while(1) {
       my($name,$value) = $qh->fetchrow_array;
       last if ! defined $name;
       push(@configs,[$name,$value]);
    };
    $qh->finish;
    $dbh->do("DROP TABLE config_old");
    $dbh->do("RENAME TABLE config TO config_old");
    $dbh->do("CREATE TABLE config (". $table{'config'} .")");
    $qh = $dbh->prepare("INSERT INTO config SET name = ?, value = ?");
    foreach my $ref (@configs) {
       $qh->execute($$ref[0],$$ref[1]);
    };
}

AddField ('user','email_pref','mediumtext not null');
AddField ('log','mod_by','int unsigned');
AddField ('log','modifyTS','int unsigned not null');

# add a changeId to tables with a _changes table
my $qh = $dbh->prepare("SELECT tablesql, pkey FROM fktable");
$qh->execute;
while(1) {  # for every 
   my($tablesql,$pkey) = $qh->fetchrow_array;
   last if ! defined $tablesql;
   my $has_change_table;
   if(TableExists($tablesql ."_change")) {
      $has_change_table = 1;
      RenameField ($tablesql .'_change','timestamp','createTS');
      RenameField ($tablesql .'_change','who','create_by');
      $dbh->do("UPDATE fktable SET changes = 1 WHERE tablesql = '$tablesql'");
      if(! GetFieldDef($tablesql, "changeId")) {  # if changeId field does not exist
         AddField ($tablesql,'changeId','int unsigned not null');
         my $changeId = fetch_one("SELECT changeId from ${tablesql}_change ORDER BY changeID DESC LIMIT 1");
         $dbh->do("UPDATE $tablesql SET changeId = $changeId");
      };
   } else {  # else the _changes table does not exist
      $dbh->do("UPDATE fktable SET changes = 0 WHERE tablesql = '$tablesql'");
      next;
   }
   if(TableExists($tablesql ."_comment")) {
      RenameField ('feature_comment','timestamp','createTS');
      RenameField ('feature_comment','who','create_by');
      AddField ('feature_comment','mod_by','int unsigned');
      AddField ('feature_comment','modifyTS','int unsigned not null');
      $dbh->do("UPDATE fktable SET comments = 1 WHERE tablesql = '$tablesql'");
      if($has_change_table && ! GetFieldDef($tablesql ."_comment", "changeId")) {  # if changeId field does not exist
         AddField ($tablesql ."_comment",'changeId','int unsigned not null');
         my $qh = $dbh->prepare("SELECT commentId, createTS, $pkey FROM ${tablesql}_comment");  # for every comment
         $qh->execute;
         while(1) { # for every comment
            my($commentId,$createTS,$featureId) = $qh->fetchrow_array;
            last if ! defined $commentId;
            my $changeId = fetch_one("SELECT changeId FROM $[tablesql}_change WHERE $pkey = $featureId AND createTS <= $createTS");
            $dbh->do("UPDATE ${tablesql}_comment SET changeId = $changeId WHERE commentId = $commentId");
         };
      }
   } else {  # else the _changes table does not exist
      $dbh->do("UPDATE fktable SET comments = 0 WHERE tablesql = '$tablesql'");
   }
}

RenameField ('fktable_comment','featureId','fktableId');
AddField ('field_def','on_edit_form','bool not null default 1');

if(AddField ('session','sessionId','int unsigned not null')) {  # really should be auto-increment
    $dbh->do("CREATE TABLE session_$$ (\n". $table{'session'} ."\n)") or die;
    $dbh->do("LOCK TABLES session_$$ WRITE, session READ") or die;
    $dbh->do("INSERT INTO session_$$ (cookie, userid, lastused, remoteip, isvalid, debug) ".
                              "SELECT cookie, userid, lastused, remoteip, isvalid, debug FROM session") or die;
    $dbh->do("UNLOCK TABLES") or die;
    $dbh->do("DROP TABLE session") or die;
    $dbh->do("ALTER TABLE session_$$ RENAME session") or die;
}

# If you had to change the --TABLE-- definition in any way, then add your
# differential change code *** A B O V E *** this comment.
#
# That is: if you add a new field, you first search for the first occurence
# of --TABLE-- and add your field to into the table hash. This new setting
# would be honored for every new installation. Then add your
# AddField/DropField/ChangeFieldType/RenameField code above. This would then
# be honored by everyone who updates his DCVS installation.
#
#
# Final checks...

print("\n".
      "Well, then.  Everything looks great.\n".
      "You might still need to add featurekong_httpd.conf to Apache.\n".
      "If you have a /etc/httpd/conf.d, simply place it in there.\n".
      "You would need to restart Apache afterwards.\n".
      "Enjoy!\n\n");




