#!/usr/bin/perl -w
use strict;
use Carp;

#
#  With this script we manipulate the database directly and
#  check some text of pages produced by list.pm and table.pm 
#  that list multiple records, but only one at a time for these tests.
#  The query stuff is untested.
#
#  To be able to find the correct column in the html listing
#  we temporarily alter the title of the field.
#

use lib ".";
use lib ".";
use lib "..";
require "../featurekong_startup.pl";
use Date::Format;
use FKong::db;

my $base_url = 'http://127.0.0.1/fkong/';

# log ourselves in (hopefully as the administrator)
my $cookie_val = 'abcdefghijklmnopqrstuv';
FKong::db::SendSQL("REPLACE session SET cookie = ". FKong::db::SqlQuote($cookie_val) .",\n".
                   "userid = 100, isvalid = 1, remoteip = 2130706433, lastused = UNIX_TIMESTAMP() ");

my $curl_cmd = "curl -s --header 'Cookie: featurekong_session=$cookie_val' ";

my $featureId = 3;
my $tablesql = 'feature';


my $dbh = FKong::db::SendSQL("SELECT pkey, url FROM fktable\n".
             "WHERE tablesql = ". FKong::db::SqlQuote($tablesql));
my($pkey, $url) = $dbh->fetchrow_array();
defined $pkey || die;
defined $url || die;

# Make sure our victim record exists
$dbh = FKong::db::SendSQL("SELECT $pkey FROM $tablesql WHERE $pkey = $featureId"); 
if(! $dbh->fetchrow_array()) { 
   FKong::db::SendSQL("INSERT INTO $tablesql SET $pkey = $featureId"); 
}

# get time format
$dbh = FKong::db::SendSQL("SELECT value FROM config WHERE name = 'ListTimeFormat'");
my($timeformat) = $dbh->fetchrow_array();
$timeformat ||= '%Y-%b-%d <nobr>%l:%M%P %Z</nobr>';

sub do_type_test
{
   my($type, undef, $db_val, $form_val) = @_;

   my $dbh = FKong::db::SendSQL("SELECT fieldId, fieldName, sqlName FROM field_def\n".
                 "WHERE tablesql = ". FKong::db::SqlQuote($tablesql) ."\n".
                 "AND type = ". FKong::db::SqlQuote($type));
   while(my($fieldId, $fieldName, $sqlName) = $dbh->fetchrow_array()) {
      my $dbv = defined $db_val ? $db_val : 'undef';
      #print "fieldId=$fieldId fieldName=$fieldName db_val=$dbv\n";
      # save old database
      my $dbh2 = FKong::db::SendSQL("SELECT $sqlName FROM $tablesql\n".   
                         "WHERE $tablesql.$pkey = $featureId"); 
      my($origValue) = $dbh2->fetchrow_array();
      # make database good for the test
      FKong::db::SendSQL("UPDATE $tablesql SET $sqlName = ". FKong::db::SqlQuote($db_val) ."\n".
                         "WHERE $tablesql.$pkey = $featureId"); 
      my $pattern = "$fieldName ". rand();
      FKong::db::SendSQL("UPDATE field_def SET fieldName = ". FKong::db::SqlQuote($pattern) ."\n".
                         "WHERE fieldId = $fieldId");
      $pattern =~ s/(\W)/\\$1/g;  # escape for regex pattern matching
      my $page = `$curl_cmd $base_url/$url/list.html?list=$featureId&allcols=1`;
      my @rows = split("<tr",$page);
      my $found;
      my $problem;
      my $col_num;
      my $right_num_cols;
      (my $pattern2 = $form_val) =~ s/([^\w\s='"<>:])/\\$1/g;  # regexp escape
      foreach my $row (@rows) {
         my @cols = split(/<t[dh]/,$row);
         if(! defined $col_num) {
            foreach (0..$#cols) {
               if($cols[$_] =~ /$pattern/) {
                  $col_num = $_;
                  last;
               }
            }
            $right_num_cols = $#cols;
            next;
         } elsif($right_num_cols == $#cols) {
            $found = 1;
            if($cols[$col_num] !~ /[^>]*>$pattern2<\/td>/) {
                warn "bad value for fieldName=$fieldName db_val=\"$dbv\" form_val=\"$form_val\"\n".
                                "pattern2=\"$pattern2\" col_num=$col_num\ncol=$cols[$col_num]\n ";
                $problem = 1;
            }
            last;
         };
      }
      # make page available for debugging
      if($problem || ! $col_num || ! $found) {
         open(my $fd,">rick.html"); 
         print $fd $page; 
         close $fd;
      };
      # repair and restore
      FKong::db::SendSQL("UPDATE field_def SET fieldName = ". FKong::db::SqlQuote($fieldName) ."\n".
                         "WHERE fieldId = $fieldId");
      FKong::db::SendSQL("UPDATE $tablesql SET $sqlName = ". FKong::db::SqlQuote($origValue) ."\n".
                         "WHERE $tablesql.$pkey = $featureId");
      $col_num or confess "field name not found for $pattern\n";
      $found or confess "data row not found for $pattern\n";
      exit(-1) if $problem;
   }
}

do_type_test('money', 0, 0,      '0.00');
do_type_test('money', 0, 1,      '0.01');
do_type_test('money', 0, 91,     '0.91');
do_type_test('money', 0, 200,    '2.00');
do_type_test('money', 0, 4000,   '40.00');
do_type_test('money', 0, -123456,'-1234.56');
do_type_test('money', 0, -3,     '-0.03');
do_type_test('money', 0, -14,    '-0.14');
do_type_test('money', 0, -800,   '-8.00');
do_type_test('money', 1, undef,  '&nbsp;');

#do_type_test('ip',    0, sprintf("%u",0x01020304), '<a href="http://1.2.3.4">1.2.3.4</a>');
#do_type_test('ip',    0, sprintf("%u",0xfffefdfc), '<a href="http://255.254.253.252">255.254.253.252</a>');
#do_type_test('ip',    0, sprintf("%u",0xfcfdfeff), '<a href="http://252.253.254.255">252.253.254.255</a>');
#do_type_test('ip',    0, 0,                        '<a href="http://0.0.0.0">0.0.0.0</a>');
#do_type_test('ip',    0, undef,                    '&nbsp;');
#do_type_test('ip',    1, sprintf("%u",0xfcfdfeff), '<a href="http://252.253.254.255">252.253.254.255</a>');
#do_type_test('oneline', 1, 'sailing.fdd.com',  '<a href="http://sailing.fdd.com">sailing.fdd.com</a>');

do_type_test('ip',    0, sprintf("%u",0x01020304), '1.2.3.4');
do_type_test('ip',    0, sprintf("%u",0xfffefdfc), '255.254.253.252');
do_type_test('ip',    0, sprintf("%u",0xfcfdfeff), '252.253.254.255');
do_type_test('ip',    0, 0,                        '0.0.0.0');
do_type_test('ip',    0, undef,                    '&nbsp;');

do_type_test('integer', 0, 0,          '0');
do_type_test('integer', 0, 1,          '1');
do_type_test('integer', 0, -1,         '-1');
do_type_test('integer', 0, 999999999,  '999999999');
do_type_test('integer', 0, -999999999, '-999999999');
do_type_test('integer', 1, undef,      '&nbsp;');
do_type_test('integer', 1, 3,          '3');

do_type_test('float', 0, 0,      '0');
do_type_test('float', 0, 1,      '1');
do_type_test('float', 0, .1,     '0.1');
do_type_test('float', 0, -.1,    '-0.1');
do_type_test('float', 0, 1.0,    '1');
do_type_test('float', 0, -1.0,   '-1');
do_type_test('float', 1, undef,  '&nbsp;');

do_type_test('hexint', 0, 15,     '0x0f');
do_type_test('hexint', 0, 1,      '0x01');
do_type_test('hexint', 0, 0,      '0x00');
do_type_test('hexint', 0, 256,    '0x100');
do_type_test('hexint', 0, sprintf("%u",0x12345678), '0x12345678');
do_type_test('hexint', 0, sprintf("%u",0xfedcba98), '0xfedcba98');
do_type_test('hexint', 1, undef,  '&nbsp;');

do_type_test('unix_ts', 0, 0,  '&nbsp;');
do_type_test('unix_ts', 1, undef,  '&nbsp;');
do_type_test('unix_ts', 0, 1000000,  time2str($timeformat,1000000));
do_type_test('unix_ts', 1, 1004000,  time2str($timeformat,1004000));

do_type_test('oneline', 1, 'blue green red',  'blue green red');
do_type_test('oneline', 1, 'blue& green red',  'blue&amp; green red');
do_type_test('oneline', 1, 'blue< green> red',  'blue&lt; green&gt; red');
do_type_test('oneline', 1, "blue\nred",  "blue\n<br>red");   # should never happen
do_type_test('oneline', 1, undef,  '&nbsp;');
do_type_test('oneline', 1, '',  '&nbsp;');

do_type_test('bool', 1, 1,  'yes');
do_type_test('bool', 1, 0,  '<font color="#888888">no</font>');

#do_type_test('multiline', 0, 'foo',  'foo');
#do_type_test('multiline', 0, '',  '&nbsp;');
#do_type_test('multiline', 0, undef,  '&nbsp;');
#do_type_test('multiline', 1, "foo\nbar",  qq{foo\n<br>bar});
#do_type_test('multiline', 1, "foo\nbar\nme",  qq{foo\n<br>bar\n<br>me});
#do_type_test('multiline', 1, "foo\nbar\nme\n",  qq{foo\n<br>bar\n<br>me\n<br>});
#do_type_test('multiline', 1, "foo\nbar\nme\n\n",  qq{foo\n<br>bar\n<br>me\n<br>\n<br>});
#do_type_test('multiline', 1, "\nfoo\nbar",  qq{\n<br>foo\n<br>bar});
#do_type_test('multiline', 1, "\n\nfoo\nbar",  qq{\n<br>\n<br>foo\n<br>bar});


#'mult_choice',

do_type_test('constant', 1, "foo",  "foo");

$dbh = FKong::db::SendSQL("SELECT userid, username FROM user LIMIT 100");
my(%userid_to_name, %username_to_id);
while(my($userid,$username) = $dbh->fetchrow_array()) {
   $userid_to_name{$userid} = $username;
   $username_to_id{$username} = $userid;
};
scalar(keys %userid_to_name) < 100 or die "too many users to run test.";
foreach my $userid (keys %userid_to_name) {
   my $name = $userid_to_name{$userid};
   # we have admin privs so usernames are a link
   do_type_test('user', 1, $userid,  "<a href=\"/fkong/user/~$name.html\">$name</a>");
}



