#!/usr/bin/perl -w
# -*- Mode: perl; indent-tabs-mode: nil -*-
#
# The contents of this file are subject to the Mozilla Public
# License Version 1.1 (the "License"); you may not use this file
# except in compliance with the License. You may obtain a copy of
# the License at http://www.mozilla.org/MPL/
#
# Software distributed under the License is distributed on an "AS
# IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
# implied. See the License for the specific language governing
# rights and limitations under the License.
#
# The Original Code is mozilla.org code.
#
# The Initial Developer of the Original Code is Holger
# Schurig. Portions created by Holger Schurig are
# Copyright (C) 1999 Holger Schurig. All
# Rights Reserved.
#
# Contributor(s): Holger Schurig <holgerschurig@nikocity.de>
#               Terry Weissman <terry@mozilla.org>
#
#
# Direct any questions on this source code to
#
# Holger Schurig <holgerschurig@nikocity.de>

use diagnostics;
use strict;

require "CGI.pl";
require "globals.pl";




# TestPlan:  just returns if the specified plan does exists
# CheckPlan: same check, optionally  emit an error text
# TestVersion:  just returns if the specified plan/version combination exists
# CheckVersion: same check, optionally emit an error text

sub TestPlan ($)
{
    my $prod = shift;

    # does the plan exist?
    SendSQL("SELECT plan
             FROM plans
             WHERE plan=" . SqlQuote($prod));
    return FetchOneColumn();
}

sub CheckPlan ($)
{
    my $prod = shift;

    # do we have a plan?
    unless ($prod) {
        print "Sorry, you haven't specified a plan.";
        PutTrailer();
        exit;
    }

    unless (TestPlan $prod) {
        print "Sorry, plan '$prod' does not exist.";
        PutTrailer();
        exit;
    }
}

sub TestVersion ($$)
{
    my ($prod,$ver) = @_;

    # does the plan exist?
    SendSQL("SELECT plan,value
             FROM versions
             WHERE plan=" . SqlQuote($prod) . " and value=" . SqlQuote($ver));
    return FetchOneColumn();
}

sub CheckVersion ($$)
{
    my ($prod,$ver) = @_;

    # do we have the version?
    unless ($ver) {
        print "Sorry, you haven't specified a version.";
        PutTrailer();
        exit;
    }

    CheckPlan($prod);

    unless (TestVersion $prod,$ver) {
        print "Sorry, version '$ver' for plan '$prod' does not exist.";
        PutTrailer();
        exit;
    }
}


#
# Displays the form to edit a version
#

sub EmitFormElements ($$)
{
    my ($plan, $version) = @_;

    print "  <TH ALIGN=\"right\">Version:</TH>\n";
    print "  <TD><INPUT SIZE=64 MAXLENGTH=64 NAME=\"version\" VALUE=\"" .
        value_quote($version) . "\">\n";
    print "      <INPUT TYPE=HIDDEN NAME=\"plan\" VALUE=\"" .
        value_quote($plan) . "\"></TD>\n";
}


#
# Displays a text like "a.", "a or b.", "a, b or c.", "a, b, c or d."
#

sub PutTrailer (@)
{
    my (@links) = ("Back to the <A HREF=\"query.cgi\">query page</A>", @_);

    my $count = $#links;
    my $num = 0;
    print "<P>\n";
    foreach (@links) {
        print $_;
        if ($num == $count) {
            print ".\n";
        }
        elsif ($num == $count-1) {
            print " or ";
        }
        else {
            print ", ";
        }
        $num++;
    }
    PutFooter();
}







#
# Preliminary checks:
#

confirm_login();

print "Content-type: text/html\n\n";

unless (UserInGroup("editcomponents")) {
    PutHeader("Not allowed");
    print "Sorry, you aren't a member of the 'editcomponents' group.\n";
    print "And so, you aren't allowed to add, modify or delete versions.\n";
    PutTrailer();
    exit;
}


#
# often used variables
#
my $plan = trim($::FORM{plan} || '');
my $version = trim($::FORM{version} || '');
my $action  = trim($::FORM{action}  || '');
my $localtrailer;
if ($version) {
    $localtrailer = "<A HREF=\"editversions.cgi?plan=" . url_quote($plan) . "\">edit</A> more versions";
} else {
    $localtrailer = "<A HREF=\"editversions.cgi\">edit</A> more versions";
}



#
# plan = '' -> Show nice list of versions
#

unless ($plan) {
    PutHeader("Select plan");

    SendSQL("SELECT plans.plan,plans.description,'xyzzy'
             FROM plans 
             GROUP BY plans.plan
             ORDER BY plans.plan");
    print "<TABLE BORDER=1 CELLPADDING=4 CELLSPACING=0><TR BGCOLOR=\"#6666FF\">\n";
    print "  <TH ALIGN=\"left\">Edit versions of ...</TH>\n";
    print "  <TH ALIGN=\"left\">Description</TH>\n";
    print "  <TH ALIGN=\"left\">Bugs</TH>\n";
    #print "  <TH ALIGN=\"left\">Edit</TH>\n";
    print "</TR>";
    while ( MoreSQLData() ) {
        my ($plan, $description, $bugs) = FetchSQLData();
        $description ||= "<FONT COLOR=\"red\">missing</FONT>";
        $bugs ||= "none";
        print "<TR>\n";
        print "  <TD VALIGN=\"top\"><A HREF=\"editversions.cgi?plan=", url_quote($plan), "\"><B>$plan</B></A></TD>\n";
        print "  <TD VALIGN=\"top\">$description</TD>\n";
        print "  <TD VALIGN=\"top\">$bugs</TD>\n";
        #print "  <TD VALIGN=\"top\"><A HREF=\"editversions.cgi?action=edit&plan=", url_quote($plan), "\">Edit</A></TD>\n";
    }
    print "</TR></TABLE>\n";

    PutTrailer();
    exit;
}



#
# action='' -> Show nice list of versions
#

unless ($action) {
    PutHeader("Select version of $plan");
    CheckPlan($plan);

=for me

    # Das geht nicht wie vermutet. Ich bekomme nicht alle Versionen
    # angezeigt!  Schade. Ich wrde gerne sehen, wieviel Bugs pro
    # Version angegeben sind ...

    SendSQL("SELECT value,plan,COUNT(bug_id)
             FROM versions LEFT JOIN bugs
               ON versions.plan=bugs.plan AND value=version
             WHERE plan=" . SqlQuote($plan) . "
             GROUP BY value");

=cut

    SendSQL("SELECT value,plan
             FROM versions 
             WHERE plan=" . SqlQuote($plan) . "
             ORDER BY value");

    print "<TABLE BORDER=1 CELLPADDING=4 CELLSPACING=0><TR BGCOLOR=\"#6666FF\">\n";
    print "  <TH ALIGN=\"left\">Edit version ...</TH>\n";
    #print "  <TH ALIGN=\"left\">Bugs</TH>\n";
    print "  <TH ALIGN=\"left\">Action</TH>\n";
    print "</TR>";
    while ( MoreSQLData() ) {
        my ($version,$dummy,$bugs) = FetchSQLData();
        $bugs ||= 'none';
        print "<TR>\n";
        print "  <TD VALIGN=\"top\"><A HREF=\"editversions.cgi?plan=", url_quote($plan), "&version=", url_quote($version), "&action=edit\"><B>$version</B></A></TD>\n";
        #print "  <TD VALIGN=\"top\">$bugs</TD>\n";
        print "  <TD VALIGN=\"top\"><A HREF=\"editversions.cgi?plan=", url_quote($plan), "&version=", url_quote($version), "&action=del\"><B>Delete</B></A></TD>\n";
        print "</TR>";
    }
    print "<TR>\n";
    print "  <TD VALIGN=\"top\">Add a new version</TD>\n";
    print "  <TD VALIGN=\"top\" ALIGN=\"middle\"><A HREF=\"editversions.cgi?plan=", url_quote($plan) . "&action=add\">Add</A></TD>\n";
    print "</TR></TABLE>\n";

    PutTrailer();
    exit;
}




#
# action='add' -> present form for parameters for new version
#
# (next action will be 'new')
#

if ($action eq 'add') {
    PutHeader("Add version of $plan");
    CheckPlan($plan);

    #print "This page lets you add a new version to a bugzilla-tracked plan.\n";

    print "<FORM METHOD=POST ACTION=editversions.cgi>\n";
    print "<TABLE BORDER=0 CELLPADDING=4 CELLSPACING=0><TR>\n";

    EmitFormElements($plan, $version);

    print "</TABLE>\n<HR>\n";
    print "<INPUT TYPE=SUBMIT VALUE=\"Add\">\n";
    print "<INPUT TYPE=HIDDEN NAME=\"action\" VALUE=\"new\">\n";
    print "</FORM>";

    my $other = $localtrailer;
    $other =~ s/more/other/;
    PutTrailer($other);
    exit;
}



#
# action='new' -> add version entered in the 'action=add' screen
#

if ($action eq 'new') {
    PutHeader("Adding new version");
    CheckPlan($plan);

    # Cleanups and valididy checks

    unless ($version) {
        print "You must enter a text for the new version. Please press\n";
        print "<b>Back</b> and try again.\n";
        PutTrailer($localtrailer);
        exit;
    }
    if (TestVersion($plan,$version)) {
        print "The version '$version' already exists. Please press\n";
        print "<b>Back</b> and try again.\n";
        PutTrailer($localtrailer);
        exit;
    }

    # Add the new version
    SendSQL("INSERT INTO versions ( " .
          "value, plan" .
          " ) VALUES ( " .
          SqlQuote($version) . "," .
          SqlQuote($plan) . ")");

    # Make versioncache flush
    unlink "data/versioncache";

    print "OK, done.<p>\n";
    PutTrailer($localtrailer);
    exit;
}




#
# action='del' -> ask if user really wants to delete
#
# (next action would be 'delete')
#

if ($action eq 'del') {
    PutHeader("Delete version of $plan");
    CheckVersion($plan, $version);

    SendSQL("SELECT count(bug_id),plan,version
             FROM bugs
             GROUP BY plan,version
             HAVING plan=" . SqlQuote($plan) . "
                AND version=" . SqlQuote($version));
    my $bugs = FetchOneColumn();

    print "<TABLE BORDER=1 CELLPADDING=4 CELLSPACING=0>\n";
    print "<TR BGCOLOR=\"#6666FF\">\n";
    print "  <TH VALIGN=\"top\" ALIGN=\"left\">Part</TH>\n";
    print "  <TH VALIGN=\"top\" ALIGN=\"left\">Value</TH>\n";

    print "</TR><TR>\n";
    print "  <TH ALIGN=\"left\" VALIGN=\"top\">Plan:</TH>\n";
    print "  <TD VALIGN=\"top\">$plan</TD>\n";
    print "</TR><TR>\n";
    print "  <TH ALIGN=\"left\" VALIGN=\"top\">Version:</TH>\n";
    print "  <TD VALIGN=\"top\">$version</TD>\n";
    print "</TR><TR>\n";
    print "  <TH ALIGN=\"left\" VALIGN=\"top\">Bugs:</TH>\n";
    print "  <TD VALIGN=\"top\">", $bugs || 'none' , "</TD>\n";
    print "</TR></TABLE>\n";

    print "<H2>Confirmation</H2>\n";

    if ($bugs) {
        if (!Param("allowbugdeletion")) {
            print "Sorry, there are $bugs features outstanding for this version.
You must reassign those features to another version before you can delete this
one.";
            PutTrailer($localtrailer);
            exit;
        }
        print "<TABLE BORDER=0 CELLPADDING=20 WIDTH=\"70%\" BGCOLOR=\"red\"><TR><TD>\n",
              "There are features entered for this version!  When you delete this ",
              "version, <B><BLINK>all</BLINK></B> stored features will be deleted, too. ",
              "You could not even see the feature history for this version anymore!\n",
              "</TD></TR></TABLE>\n";
    }

    print "<P>Do you really want to delete this version?<P>\n";
    print "<FORM METHOD=POST ACTION=editversions.cgi>\n";
    print "<INPUT TYPE=SUBMIT VALUE=\"Yes, delete\">\n";
    print "<INPUT TYPE=HIDDEN NAME=\"action\" VALUE=\"delete\">\n";
    print "<INPUT TYPE=HIDDEN NAME=\"plan\" VALUE=\"" .
        value_quote($plan) . "\">\n";
    print "<INPUT TYPE=HIDDEN NAME=\"version\" VALUE=\"" .
        value_quote($version) . "\">\n";
    print "</FORM>";

    PutTrailer($localtrailer);
    exit;
}



#
# action='delete' -> really delete the version
#

if ($action eq 'delete') {
    PutHeader("Deleting version of $plan");
    CheckVersion($plan,$version);

    # lock the tables before we start to change everything:

    SendSQL("LOCK TABLES attachments WRITE,
                         bugs WRITE,
                         bugs_activity WRITE,
                         versions WRITE,
                         dependencies WRITE");

    # According to MySQL doc I cannot do a DELETE x.* FROM x JOIN Y,
    # so I have to iterate over bugs and delete all the indivial entries
    # in bugs_activies and attachments.

    if (Param("allowbugdeletion")) {

        SendSQL("SELECT bug_id
             FROM bugs
             WHERE plan=" . SqlQuote($plan) . "
               AND version=" . SqlQuote($version));
        while (MoreSQLData()) {
            my $bugid = FetchOneColumn();

            PushGlobalSQLState();
            SendSQL("DELETE FROM attachments WHERE bug_id=$bugid");
            SendSQL("DELETE FROM bugs_activity WHERE bug_id=$bugid");
            SendSQL("DELETE FROM dependencies WHERE blocked=$bugid");
            PopGlobalSQLState();
        }
        print "Attachments, bug activity and dependencies deleted.<BR>\n";


        # Deleting the rest is easier:

        SendSQL("DELETE FROM bugs
             WHERE plan=" . SqlQuote($plan) . "
               AND version=" . SqlQuote($version));
        print "Bugs deleted.<BR>\n";
    }

    SendSQL("DELETE FROM versions
             WHERE plan=" . SqlQuote($plan) . "
               AND value=" . SqlQuote($version));
    print "Version deleted.<P>\n";
    SendSQL("UNLOCK TABLES");

    unlink "data/versioncache";
    PutTrailer($localtrailer);
    exit;
}



#
# action='edit' -> present the edit version form
#
# (next action would be 'update')
#

if ($action eq 'edit') {
    PutHeader("Edit version of $plan");
    CheckVersion($plan,$version);

    print "<FORM METHOD=POST ACTION=editversions.cgi>\n";
    print "<TABLE BORDER=0 CELLPADDING=4 CELLSPACING=0><TR>\n";

    EmitFormElements($plan, $version);

    print "</TR></TABLE>\n";

    print "<INPUT TYPE=HIDDEN NAME=\"versionold\" VALUE=\"" .
        value_quote($version) . "\">\n";
    print "<INPUT TYPE=HIDDEN NAME=\"action\" VALUE=\"update\">\n";
    print "<INPUT TYPE=SUBMIT VALUE=\"Update\">\n";

    print "</FORM>";

    my $other = $localtrailer;
    $other =~ s/more/other/;
    PutTrailer($other);
    exit;
}



#
# action='update' -> update the version
#

if ($action eq 'update') {
    PutHeader("Update version of $plan");

    my $versionold = trim($::FORM{versionold} || '');

    CheckVersion($plan,$versionold);

    # Note that the order of this tests is important. If you change
    # them, be sure to test for WHERE='$version' or WHERE='$versionold'

    SendSQL("LOCK TABLES bugs WRITE,
                         versions WRITE");

    if ($version ne $versionold) {
        unless ($version) {
            print "Sorry, I can't delete the version text.";
            PutTrailer($localtrailer);
	    SendSQL("UNLOCK TABLES");
            exit;
        }
        if (TestVersion($plan,$version)) {
            print "Sorry, version '$version' is already in use.";
            PutTrailer($localtrailer);
	    SendSQL("UNLOCK TABLES");
            exit;
        }
        SendSQL("UPDATE bugs
                 SET version=" . SqlQuote($version) . "
                 WHERE version=" . SqlQuote($versionold) . "
                   AND plan=" . SqlQuote($plan));
        SendSQL("UPDATE versions
                 SET value=" . SqlQuote($version) . "
                 WHERE plan=" . SqlQuote($plan) . "
                   AND value=" . SqlQuote($versionold));
        unlink "data/versioncache";
        print "Updated version.<BR>\n";
    }
    SendSQL("UNLOCK TABLES");

    PutTrailer($localtrailer);
    exit;
}



#
# No valid action found
#

PutHeader("Error");
print "I don't have a clue what you want.<BR>\n";

foreach ( sort keys %::FORM) {
    print "$_: $::FORM{$_}<BR>\n";
}
