#!/usr/bin/perl -w
# -*- Mode: perl; indent-tabs-mode: nil -*-
#
# The contents of this file are subject to the Mozilla Public
# License Version 1.1 (the "License"); you may not use this file
# except in compliance with the License. You may obtain a copy of
# the License at http://www.mozilla.org/MPL/
#
# Software distributed under the License is distributed on an "AS
# IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
# implied. See the License for the specific language governing
# rights and limitations under the License.
#
# The Original Code is the Bugzilla Bug Tracking System.
#
# The Initial Developer of the Original Code is Netscape Communications
# Corporation. Portions created by Netscape are
# Copyright (C) 1998 Netscape Communications Corporation. All
# Rights Reserved.
#
# Contributor(s): Terry Weissman <terry@mozilla.org>
#                 Dan Mosedale <dmose@mozilla.org>
#                 Dave Miller <justdave@syndicomm.com>
#                 Christopher Aillon <christopher@aillon.com>
#                 Rick Dean <software@fdd.com>

use diagnostics;
use strict;

my $UserInEditGroupSet = -1;

require "CGI.pl";
use RelationSet;

# Shut up misguided -w warnings about "used only once":

use vars %::versions,
    %::components,
    %::COOKIE,
    %::MFORM,
    %::legal_keywords,
    %::legal_sponsor,
    %::legal_status,
    %::legal_committee,
    %::target_milestone,
    %::legal_severity;

my $whoid = confirm_login();

my $requiremilestone = 0;

######################################################################
# Begin Data/Security Validation
######################################################################

# Create a list of IDs of all bugs being modified in this request.
# This list will either consist of a single feature number from the "id"
# form/URL field or a series of numbers from multiple form/URL fields
# named "id_x" where "x" is the feature number.
my @idlist;
if (defined $::FORM{'id'}) {
    push @idlist, $::FORM{'id'};
} else {
    foreach my $i (keys %::FORM) {
        if ($i =~ /^id_([1-9][0-9]*)/) {
            push @idlist, $1;
        }
    }
}

#print("content-type:text/html\n\n");
#for my $key (keys(%::FORM)) {
#   print("<h1>$key -> $::FORM{$key}</h1>\n");
#}


# For each feature being modified, make sure its ID is a valid feature number 
# representing an existing feature that the user is authorized to access.
foreach my $id (@idlist) {
    ValidateBugID($id);
}

# If we are duping bugs, let's also make sure that we can change 
# the original.  This takes care of issue A on feature 96085.
if ($::FORM{'dup_id'}) {
    ValidateBugID($::FORM{'dup_id'});

    # Also, let's see if the reporter has authorization to see the bug
    # to which we are duping.  If not we need to prompt.
    DuplicateUserConfirm();
}

# If the user has a feature list and is processing one bug, then after
# we process the feature we are going to show them the next feature on their
# list.  Thus we have to make sure this feature ID is also valid,
# since a malicious cracker might alter their cookies for the purpose
# gaining access to bugs they are not authorized to access.
if ( defined $::COOKIE{"BUGLIST"} && $::COOKIE{"BUGLIST"} ne "" && defined $::FORM{'id'} ) {
    my @buglist = split( /:/ , $::COOKIE{"BUGLIST"} );
    my $idx = lsearch( \@buglist , $::FORM{"id"} );
    if ($idx < $#buglist) {
        my $nextbugid = $buglist[$idx + 1];
        ValidateBugID($nextbugid);
    }
}

######################################################################
# End Data/Security Validation
######################################################################

print "Content-type: text/html\n\n";

PutHeader ("Feature processed");

GetVersionTable();

if ( Param("strictvaluechecks") ) {
    CheckFormFieldDefined(\%::FORM, 'plan');
    if (Param("useversion")) {
       CheckFormFieldDefined(\%::FORM, 'version');
    };
    CheckFormFieldDefined(\%::FORM, 'component');

    # check if target milestone is defined - matthew@zeroknowledge.com
    if ( Param("usetargetmilestone") ) {
        CheckFormFieldDefined(\%::FORM, 'target_milestone');
    }
}

ConnectToDatabase();

# Figure out whether or not the user is trying to change the plan
# (either the "plan" variable is not set to "don't change" or the
# user is changing a single feature and has changed the bug's plan),
# and make the user verify the version, component, target milestone,
# and feature groups if so.
if ( $::FORM{'id'} ) {
    SendSQL("SELECT plan FROM bugs WHERE bug_id = $::FORM{'id'}");
    $::oldplan = FetchSQLData();
}
if ( ($::FORM{'id'} && $::FORM{'plan'} ne $::oldplan) 
       || (!$::FORM{'id'} && $::FORM{'plan'} ne $::dontchange) ) {
    if ( Param("strictvaluechecks") ) {
        CheckFormField(\%::FORM, 'plan', \@::legal_plan);
    }
    my $prod = $::FORM{'plan'};

    # note that when this script is called from buglist.cgi (rather
    # than show_bug.cgi), it's possible that the plan will be changed
    # but that the version and/or component will be set to 
    # "--dont_change--" but still happen to be correct.  in this case,
    # the if statement will incorrectly trigger anyway.  this is a 
    # pretty weird case, and not terribly unreasonable behavior, but 
    # worthy of a comment, perhaps.
    #
    my $vok = lsearch($::versions{$prod}, $::FORM{'version'}) >= 0;
    my $cok = lsearch($::components{$prod}, $::FORM{'component'}) >= 0;

    my $mok = 1;   # so it won't affect the 'if' statement if milestones aren't used
    if ( Param("usetargetmilestone") ) {
       $mok = lsearch($::target_milestone{$prod}, $::FORM{'target_milestone'}) >= 0;
    }

    # If anything needs to be verified, generate a form for verifying it.
    if (!$vok || !$cok || !$mok || (Param('usebuggroups') && !defined($::FORM{'addtonewgroup'}))) {

        # Start the form.
        print qq|<form action="process_bug.cgi" method="post">\n|;

        # Add all form fields to the form as hidden fields (except those 
        # being verified), so the user's changes are preserved.
        foreach my $i (keys %::FORM) {
            if ($i ne 'version' && $i ne 'component' && $i ne 'target_milestone') {
                print qq|<input type="hidden" name="$i" value="| . value_quote($::FORM{$i}) . qq|">\n|;
            }
        }

        # Display UI for verifying the version, component, and target milestone fields.
        if (!$vok || !$cok || !$mok) {
            my ($sectiontitle, $sectiondescription);
            if ( Param('usetargetmilestone') ) {
                $sectiontitle = "Verify Version, Component, Target Release";
                $sectiondescription = qq|
                  You are moving the bug(s) to the plan <b>$prod</b>, and now the 
                  version, component, and/or target milestone fields are not correct 
                  (or perhaps they were not correct in the first place).  In any case, 
                  please set the correct version, component, and target milestone now:
                |;
            } else {
                $sectiontitle = "Verify Version, Component";
                $sectiondescription = qq|
                  You are moving the bug(s) to the plan <b>$prod</b>, and now the 
                  version, and component fields are not correct (or perhaps they were 
                  not correct in the first place).  In any case, please set the correct 
                  version and component now:
                |;
            }

            my $versionmenu = Version_element($::FORM{'version'}, $prod);
            my $componentmenu = Component_element($::FORM{'component'}, $prod);

            print qq|
              <h3>$sectiontitle</h3>

              <p>
                $sectiondescription
              <p>

              <table><tr>
              <td>
                <b>Version:</b><br>
                $versionmenu
              </td>
              <td>
                <b>Component:</b><br>
                $componentmenu
              </td>
            |;

            if ( Param("usetargetmilestone") ) {
                my $milestonemenu = Milestone_element($::FORM{'target_milestone'}, $prod);
                print qq|
                  <td>
                    <b>Target Release:</b><br>
                    $milestonemenu
                  </td>
                |;
            }

            print qq|
              </tr></table>
            |;
        }

        # Display UI for determining whether or not to remove the feature from 
        # its old plan's group and/or add it to its new plan's group.
        if (Param('usebuggroups') && !defined($::FORM{'addtonewgroup'})) {
            print qq|
              <h3>Verify Feature Group</h3>

              <p>
                Do you want to add the feature to its new plan's group (if any)?
              </p>

              <p>
                <input type="radio" name="addtonewgroup" value="no"><b>no</b><br>
                <input type="radio" name="addtonewgroup" value="yes"><b>yes</b><br>
                <input type="radio" name="addtonewgroup" value="yesifinold" checked>
                  <b>yes, but only if the feature was in its old plan's group</b><br>
              </p>
            |;
        }

        # End the form.
        print qq|
          <input type="submit" value="Commit">
          </form>
          <hr>
          <a href="query.cgi">Cancel and Return to the Query Page</a>
        |;

        # End the page and stop processing.
        PutFooter();
        exit;
    }
}


# Checks that the user is allowed to change the given field.  Actually, right
# now, the rules are pretty simple, and don't look at the field itself very
# much, but that could be enhanced.

my $lastbugid = 0;
my $ownerid;
my $reporterid;
my $qacontactid;

sub CheckCanChangeField {
    my ($f, $bugid, $oldvalue, $newvalue) = (@_);
    if ($f eq "assigned_to" || $f eq "reporter" || $f eq "qa_contact") {
        if ($oldvalue =~ /^\d+$/) {
            if ($oldvalue == 0) {
                $oldvalue = "";
            } else {
                $oldvalue = DBID_to_name($oldvalue);
            }
        }
    }
    if ($oldvalue eq $newvalue) {
        return 1;
    }
    if (trim($oldvalue) eq trim($newvalue)) {
        return 1;
    }
    if ($f =~ /^longdesc/) {
        return 1;
    }
    if ($UserInEditGroupSet < 0) {
        $UserInEditGroupSet = UserInGroup("editbugs");
    }
    if ($UserInEditGroupSet) {
        return 1;
    }
    if ($lastbugid != $bugid) {
        SendSQL("SELECT reporter, assigned_to, qa_contact FROM bugs " .
                "WHERE bug_id = $bugid");
        ($reporterid, $ownerid, $qacontactid) = (FetchSQLData());
    }
    # Let reporter change feature status, even if they can't edit bugs.
    # If reporter can't re-open their feature they will just file a duplicate.
    # While we're at it, let them close their own bugs as well.
    if ( ($f eq "status") && ($whoid eq $reporterid) ) {
        return 1;
    }
    if ($reporterid eq $whoid || $ownerid eq $whoid ||
             $qacontactid eq $whoid) {
        return 1;
    }
    SendSQL("UNLOCK TABLES");
    $oldvalue = value_quote($oldvalue);
    $newvalue = value_quote($newvalue);
    print PuntTryAgain(qq{
Only the owner or submitter of the bug, or a sufficiently
empowered user, may make that change to the $f field.
<TABLE>
<TR><TH ALIGN="right">Old value:</TH><TD>$oldvalue</TD></TR>
<TR><TH ALIGN="right">New value:</TH><TD>$newvalue</TD></TR>
</TABLE>
});
    PutFooter();
    exit();
}

# Confirm that the reporter of the current feature can access the feature we are duping to.
sub DuplicateUserConfirm {
    my $dupe = trim($::FORM{'id'});
    my $original = trim($::FORM{'dup_id'});
    
    SendSQL("SELECT reporter FROM bugs WHERE bug_id = " . SqlQuote($dupe));
    my $reporter = FetchOneColumn();
    SendSQL("SELECT profiles.groupset FROM profiles WHERE profiles.userid =".SqlQuote($reporter));
    my $reportergroupset = FetchOneColumn();

    SendSQL("SELECT ((groupset & $reportergroupset) = groupset) , reporter , assigned_to , qa_contact , 
                    reporter_accessible , assignee_accessible , qacontact_accessible , cclist_accessible 
             FROM   bugs 
             WHERE  bug_id = $original");

    my ($isauthorized, $originalreporter, $assignee, $qacontact, $reporter_accessible, 
        $assignee_accessible, $qacontact_accessible, $cclist_accessible) = FetchSQLData();

    # If reporter is authorized via the database, or is the original reporter, assignee,
    # or QA Contact, we'll automatically confirm they can be added to the cc list
    if ($isauthorized 
        || ($reporter_accessible && $originalreporter == $reporter)
          || ($assignee_accessible && $assignee == $reporter)
            || ($qacontact_accessible && $qacontact == $reporter)) {
            
        $::FORM{'confirm_add_duplicate'} = "1";
        return;    
    }

    # Try to authorize the user one more time by seeing if they are on 
    # the cc: list.  If so, finish validation and return.
    if ($cclist_accessible ) {
        my @cclist;
        SendSQL("SELECT cc.who 
                 FROM   bugs , cc
                 WHERE  bugs.bug_id = $original
                 AND    cc.bug_id = bugs.bug_id
                ");
        while (my ($ccwho) = FetchSQLData()) {
            if ($reporter == $ccwho) {
                $::FORM{'confirm_add_duplicate'} = "1";
                return;
            }
        }
    }

    if (defined $::FORM{'confirm_add_duplicate'}) {
        return;
    }
    
    # Once in this part of the subroutine, the user has not been auto-validated
    # and the duper has not chosen whether or not to add to CC list, so let's
    # ask the duper what he/she wants to do.
    
    # First, will the user gain access to this feature immediately by being CC'd?
    my $reporter_access = $cclist_accessible ? "will immediately" : "might, in the future,";

    print "Content-type: text/html\n\n";
    PutHeader("Duplicate Warning");
    print "<P>
When marking a feature as a duplicate, the reporter of the 
duplicate is normally added to the CC list of the original. 
The permissions on feature #$original (the original) are currently set 
such that the reporter would not normally be able to see it. 
<P><B>Adding the reporter to the CC list of feature #$original 
$reporter_access allow him/her access to view this bug.</B>
Do you wish to do this?</P>
</P>
";
    print "<form method=post>\n\n";

    foreach my $i (keys %::FORM) {
        # Make sure we don't include the username/password fields in the
        # HTML.  If cookies are off, they'll have to reauthenticate after
        # hitting "submit changes anyway".
        # see http://bugzilla.mozilla.org/show_bug.cgi?id=15980
        if ($i !~ /^(Bugzilla|LDAP)_(login|password)$/) {
            my $value = value_quote($::FORM{$i});
            print qq{<input type=hidden name="$i" value="$value">\n};
        }
    }

    print qq{<p><input type=radio name="confirm_add_duplicate" value="1"> Yes, add the reporter to CC list on feature $original</p>\n};
    print qq{<p><input type=radio name="confirm_add_duplicate" value="0" checked="checked"> No, do not add the reporter to CC list on feature $original</p>\n};
    print qq{\n<p><a href="show_bug.cgi?id=$dupe">Throw away my changes, and go revisit feature $dupe</a>\n};
    print qq{\n<p><input type="submit" value="Submit"></p></form>\n};
    PutFooter();
    exit;
} # end DuplicateUserConfirm()

if (defined $::FORM{'id'} && Param('strictvaluechecks')) {
    # since this means that we were called from show_bug.cgi, now is a good
    # time to do a whole bunch of error checking that can't easily happen when
    # we've been called from buglist.cgi, because buglist.cgi only tweaks
    # values that have been changed instead of submitting all the new values.
    # (XXX those error checks need to happen too, but implementing them 
    # is more work in the current architecture of this script...)
    #
    CheckFormField(\%::FORM, 'committee', \@::legal_committee);
    CheckFormField(\%::FORM, 'bug_severity', \@::legal_severity);
    CheckFormField(\%::FORM, 'component', 
                   \@{$::components{$::FORM{'plan'}}});
    CheckFormFieldDefined(\%::FORM, 'bug_file_loc');
    CheckFormFieldDefined(\%::FORM, 'short_desc');
    CheckFormField(\%::FORM, 'plan', \@::legal_plan);
    CheckFormField(\%::FORM, 'version', 
                   \@{$::versions{$::FORM{'plan'}}});
    CheckFormField(\%::FORM, 'sponsor', \@::legal_sponsor);
    CheckFormFieldDefined(\%::FORM, 'longdesclength');
    CheckFormField(\%::FORM, 'status', \@::legal_status);
    CheckFormFieldDefined(\%::FORM, 'priority');
}

my $action  = '';
if (defined $::FORM{action}) {
  $action  = trim($::FORM{action});
}
if ($action eq Param("move-button-text")) {
  $::FORM{'buglist'} = join (":", @idlist);
  do "move.pl" || die "Error executing move.cgi: $!";
  PutFooter();
  exit;
}


if (!defined $::FORM{'who'}) {
    $::FORM{'who'} = $::COOKIE{'FeatureKong_login'};
}

# the common updates to all bugs in @idlist start here
#
print "<TITLE>Update Feature " . join(" ", @idlist) . "</TITLE>\n";
if (defined $::FORM{'id'}) {
    navigation_header();
}
print "<HR>\n";
$::query = "update bugs\nset";
$::comma = "";
umask(0);

sub DoComma {
    $::query .= "$::comma\n    ";
    $::comma = ",";
}

sub ChangeStatus {
    my ($str) = (@_);
    if ($str ne $::dontchange) {
        DoComma();
        $::query .= "status = '$str'";
        $::FORM{'status'} = $str; # Used later for call to
                                      # CheckCanChangeField to make sure this
                                      # is really kosher.
    }
}

#
# This function checks if there is a comment required for a specific
# function and tests, if the comment was given.
# If comments are required for functions  is defined by params.
#
sub CheckonComment( $ ) {
    my ($function) = (@_);
    
    # Param is 1 if comment should be added !
    my $ret = Param( "commenton" . $function );

    # Allow without comment in case of undefined Params.
    $ret = 0 unless ( defined( $ret ));

    if( $ret ) {
        if (!defined $::FORM{'comment'} || $::FORM{'comment'} =~ /^\s*$/) {
            # No comment - sorry, action not allowed !
            PuntTryAgain("You have to specify a <b>comment</b> on this " .
                         "change.  Please give some words " .
                         "on the reason for your change.");
        } else {
            $ret = 0;
        }
    }
    return( ! $ret ); # Return val has to be inverted
}

# Changing this so that it will process groups from checkboxes instead of
# select lists.  This means that instead of looking for the bit-X values in
# the form, we need to loop through all the feature groups this user has access
# to, and for each one, see if it's selected.
# In addition, adding a little extra work so that we don't clobber groupsets
# for bugs where the user doesn't have access to the group, but does to the
# feature (as with the proposed reporter access patch.)
if($::usergroupset ne '0') {
  # We want to start from zero and build up, since if all boxes have been
  # unchecked, we want to revert to 0.
  DoComma();
  $::query .= "groupset = 0";
  my ($id) = (@idlist);
  SendSQL(<<_EOQ_);
    SELECT bit, bit & $::usergroupset != 0, bit & bugs.groupset != 0
    FROM groups, bugs
    WHERE isbuggroup != 0 AND bug_id = $id
    ORDER BY bit
_EOQ_
  while (my ($b, $userhasgroup, $bughasgroup) = FetchSQLData()) {
    if (!$::FORM{"bit-$b"}) {
      # If we make it here, the item didn't exist on the form or the user
      # said to clear it.  The only time we add this group back in is if
      # the feature already has this group on it and the user can't access it.
      if ($bughasgroup && !$userhasgroup) {
        $::query .= " + $b";
      }
    } elsif ($::FORM{"bit-$b"} == -1) {
      # If we get here, the user came from the change several bugs form, and
      # said not to change this group restriction.  So we'll add this group
      # back in only if the feature already has it.
      if ($bughasgroup) {
        $::query .= " + $b";
      }
    } else {
      # If we get here, the user said to set this group.  If they don't have
      # access to it, we'll use what's already on the bug, otherwise we'll
      # add this one in.
      if ($userhasgroup || $bughasgroup) {
        $::query .= " + $b";
      }
    }
  }
}

foreach my $field ("committee", "priority", "bug_severity",          
                   "summary", "component", "bug_file_loc", "short_desc",
                   "plan", "version", "sponsor", "big_desc",
                   "target_milestone", "status_whiteboard") {
    if (defined $::FORM{$field}) {
        if ($::FORM{$field} ne $::dontchange) {
            DoComma();
            $::query .= "$field = " . SqlQuote(trim($::FORM{$field}));
        }
    }
}


if (defined $::FORM{'qa_contact'}) {
    my $name = trim($::FORM{'qa_contact'});
    if ($name ne $::dontchange) {
        my $id = 0;
        if ($name ne "") {
            $id = DBNameToIdAndCheck($name);
        }
        DoComma();
        $::query .= "qa_contact = $id";
    }
}


# If the user is submitting changes from show_bug.cgi for a single bug,
# and that feature is restricted to a group, process the checkboxes that
# allowed the user to set whether or not the reporter, assignee, QA contact, 
# and cc list can see the feature even if they are not members of all groups 
# to which the feature is restricted.
if ( $::FORM{'id'} ) {
    SendSQL("SELECT groupset FROM bugs WHERE bug_id = $::FORM{'id'}");
    my ($groupset) = FetchSQLData();
    if ( $groupset ) {
        DoComma();
        $::FORM{'reporter_accessible'} = $::FORM{'reporter_accessible'} ? '1' : '0';
        $::query .= "reporter_accessible = $::FORM{'reporter_accessible'}";

        DoComma();
        $::FORM{'assignee_accessible'} = $::FORM{'assignee_accessible'} ? '1' : '0';
        $::query .= "assignee_accessible = $::FORM{'assignee_accessible'}";

        DoComma();
        $::FORM{'qacontact_accessible'} = $::FORM{'qacontact_accessible'} ? '1' : '0';
        $::query .= "qacontact_accessible = $::FORM{'qacontact_accessible'}";

        DoComma();
        $::FORM{'cclist_accessible'} = $::FORM{'cclist_accessible'} ? '1' : '0';
        $::query .= "cclist_accessible = $::FORM{'cclist_accessible'}";
    }
}

my $oldpriority = $::FORM{'priority'};
$::FORM{'priority'} =~ s/.*?(\d+).*/$1/;  # keep only integer
if($::FORM{'priority'} !~ /^\d+$/) {  # if not integerlike
    $::FORM{'priority'} = Param("defaultpriority");
};
if($::FORM{'priority'} ne $oldpriority) {
   print("<font color=red>Warning: priority changed from $oldpriority to $::FORM{'priority'}</font><br>\n");
}

my $duplicate = 0;

# We need to check the addresses involved in a CC change before we touch any bugs.
# What we'll do here is formulate the CC data into two hashes of ID's involved
# in this CC change.  Then those hashes can be used later on for the actual change.
my (%cc_add, %cc_remove);
if (defined $::FORM{newcc} || defined $::FORM{removecc} || defined $::FORM{masscc}) {
    # If masscc is defined, then we came from buglist and need to either add or
    # remove cc's... otherwise, we came from bugform and may need to do both.
    my ($cc_add, $cc_remove) = "";
    if (defined $::FORM{masscc}) {
        if ($::FORM{ccaction} eq 'add') {
            $cc_add = $::FORM{masscc};
        } elsif ($::FORM{ccaction} eq 'remove') {
            $cc_remove = $::FORM{masscc};
        }
    } else {
        $cc_add = $::FORM{newcc};
        # We came from bug_form which uses a select box to determine what cc's
        # need to be removed...
        if (defined $::FORM{removecc}) {
            $cc_remove = join (",", @{$::MFORM{cc}});
        }
    }

    if ($cc_add) {
        $cc_add =~ s/[\s,]+/ /g; # Change all delimiters to a single space
        foreach my $person ( split(" ", $cc_add) ) {
            my $pid = DBNameToIdAndCheck($person);
            $cc_add{$pid} = $person;
        }
    }
    if ($cc_remove) {
        $cc_remove =~ s/[\s,]+/ /g; # Change all delimiters to a single space
        foreach my $person ( split(" ", $cc_remove) ) {
            my $pid = DBNameToIdAndCheck($person);
            $cc_remove{$pid} = $person;
        }
    }
}

SendSQL("SELECT status FROM bugs WHERE bug_id = " . SqlQuote($::FORM{'id'}));
my $old_status = FetchOneColumn();
if($::FORM{'dup_id'} || $::FORM{'status'} =~ /^Duplicate$/i) {
    if($old_status !~ /^duplicate$/i) {
        ChangeStatus('Duplicate');
        my $num = trim($::FORM{'dup_id'});
        SendSQL("SELECT bug_id FROM bugs WHERE bug_id = " . SqlQuote($num));
        $num = FetchOneColumn();
        if (!$num) {
            PuntTryAgain("You must specify a valid feature number of which this feature " .
                         "is a duplicate.  The feature has not been changed.")
        }
        if (!defined($::FORM{'id'}) || $num == $::FORM{'id'}) {
            PuntTryAgain("Nice try, $::FORM{'who'}.  But it doesn't really ".
                         "make sense to mark a feature as a duplicate of " .
                         "itself, does it?");
        }
        my $checkid = trim($::FORM{'id'});
        SendSQL("SELECT bug_id FROM bugs where bug_id = " .  SqlQuote($checkid));
        $checkid = FetchOneColumn();
        if (!$checkid) {
            PuntTryAgain("The feature id $::FORM{'id'} is invalid. Please reload this feature ".
                         "and try again.");
        }
        $::FORM{'comment'} .= "\n\n*** This feature has been marked as a duplicate of $num ***";
        $duplicate = $num;
    };
} else {
    ChangeStatus($::FORM{'status'});
    if($old_status =~ /^duplicate$/i) {
        SendSQL("DELETE FROM duplicates WHERE dupe = $::FORM{'id'}");
    };
};

if($::FORM{'assigned_to'}) {
   if ( trim($::FORM{'assigned_to'}) eq "") {
        PuntTryAgain("You cannot reassign to a feature to nobody.  Unless " .
                     "you intentionally cleared out the " .
                     "\"Reassign feature to\" field, " .
                     Param("browserbugmessage"));
    }
    my $newid = DBNameToIdAndCheck($::FORM{'assigned_to'});
    DoComma();
    $::query .= "assigned_to = $newid";
}

if ($#idlist < 0) {
    PuntTryAgain("You apparently didn't choose any bugs to modify.");
}

my @keywordlist;
my %keywordseen;

if ($::FORM{'keywords'}) {
    foreach my $keyword (split(/[\s,]+/, $::FORM{'keywords'})) {
        if ($keyword eq '') {
            next;
        }
        my $i = GetKeywordIdFromName($keyword);
        if (!$i) {
            PuntTryAgain("Unknown keyword named <code>" .
                         html_quote($keyword) . "</code>. " .
                         "<P>The legal keyword names are " .
                         "<A HREF=describekeywords.cgi>" .
                         "listed here</A>.");
        }
        if (!$keywordseen{$i}) {
            push(@keywordlist, $i);
            $keywordseen{$i} = 1;
        }
    }
}

my $keywordaction = $::FORM{'keywordaction'} || "makeexact";

if ($::comma eq "" && 0 == @keywordlist && $keywordaction ne "makeexact") {
    if (!defined $::FORM{'comment'} || $::FORM{'comment'} =~ /^\s*$/) {
        PuntTryAgain("Um, you apparently did not change anything on the " .
                     "selected bugs.");
    }
}

my $basequery = $::query;
my $delta_ts;


sub SnapShotBug {
    my ($id) = (@_);
    SendSQL("select delta_ts, " . join(',', @::log_columns) .
            " from bugs where bug_id = $id");
    my @row = FetchSQLData();
    $delta_ts = shift @row;

    return @row;
}


sub SnapShotDeps {
    my ($i, $target, $me) = (@_);
    SendSQL("select $target from dependencies where $me = $i order by $target");
    my @list;
    while (MoreSQLData()) {
        push(@list, FetchOneColumn());
    }
    return join(',', @list);
}


my $timestamp;

sub FindWrapPoint {
    my ($string, $startpos) = @_;
    if (!$string) { return 0 }
    if (length($string) < $startpos) { return length($string) }
    my $wrappoint = rindex($string, ",", $startpos); # look for comma
    if ($wrappoint < 0) {  # can't find comma
        $wrappoint = rindex($string, " ", $startpos); # look for space
        if ($wrappoint < 0) {  # can't find space
            $wrappoint = rindex($string, "-", $startpos); # look for hyphen
            if ($wrappoint < 0) {  # can't find hyphen
                $wrappoint = $startpos;  # just truncate it
            } else {
                $wrappoint++; # leave hyphen on the left side
            }
        }
    }
    return $wrappoint;
}

sub LogActivityEntry {
    my ($i,$col,$removed,$added) = @_;
    # in the case of CCs, deps, and keywords, there's a possibility that someone
    # might try to add or remove a lot of them at once, which might take more
    # space than the activity table allows.  We'll solve this by splitting it
    # into multiple entries if it's too long.
    if($col eq "big_desc") {
        SendSQL("SELECT count(thetext) FROM big_desc_hist " .
                "WHERE bug_id = $i ");
        my $num_big_desc = FetchOneColumn();
        SendSQL("INSERT INTO big_desc_hist " .
                "(bug_id,who,bug_when,thetext,version) VALUES " .
                "($i,$whoid,$timestamp,".SqlQuote($added).",$num_big_desc)");
        $added = "description-" . ($num_big_desc);
        $removed = "description-" . ($num_big_desc - 1);
    }
    while ($removed || $added) {
        my ($removestr, $addstr) = ($removed, $added);
        if (length($removestr) > 254) {
            my $commaposition = FindWrapPoint($removed, 254);
            $removestr = substr($removed,0,$commaposition);
            $removed = substr($removed,$commaposition);
            $removed =~ s/^[,\s]+//; # remove any comma or space
        } else {
            $removed = ""; # no more entries
        }
        if (length($addstr) > 254) {
            my $commaposition = FindWrapPoint($added, 254);
            $addstr = substr($added,0,$commaposition);
            $added = substr($added,$commaposition);
            $added =~ s/^[,\s]+//; # remove any comma or space
        } else {
            $added = ""; # no more entries
        }
        $addstr = SqlQuote($addstr);
        $removestr = SqlQuote($removestr);
        my $fieldid = GetFieldID($col);
        SendSQL("INSERT INTO bugs_activity " .
                "(bug_id,who,bug_when,fieldid,removed,added) VALUES " .
                "($i,$whoid,$timestamp,$fieldid,$removestr,$addstr)");
    }
}

sub LogDependencyActivity {
    my ($i, $oldstr, $target, $me) = (@_);
    my $newstr = SnapShotDeps($i, $target, $me);
    if ($oldstr ne $newstr) {
        # Figure out what's really different...
        my ($removed, $added) = DiffStrings($oldstr, $newstr);
        LogActivityEntry($i,$target,$removed,$added);
        return 1;
    }
    return 0;
}

# this loop iterates once for each feature to be processed (eg when this script
# is called with multiple bugs selected from buglist.cgi instead of
# show_bug.cgi).
#
foreach my $id (@idlist) {
    my %dependencychanged;
    my $write = "WRITE";        # Might want to make a param to control
                                # whether we do LOW_PRIORITY ...
    SendSQL("LOCK TABLES bugs $write, bugs_activity $write, cc $write, " .
            "profiles $write, dependencies $write, votes $write, " .
            "keywords $write, longdescs $write, fielddefs $write, " .
            "keyworddefs READ, groups READ, attachments READ, plans READ, ".
            "buckets $write, big_desc_hist $write");
    my @oldvalues = SnapShotBug($id);
    my %oldhash;
    my $i = 0;
    foreach my $col (@::log_columns) {
        $oldhash{$col} = $oldvalues[$i];
        if (exists $::FORM{$col}) {
            CheckCanChangeField($col, $id, $oldvalues[$i], $::FORM{$col});
        }
        $i++;
    }
    if ($requiremilestone) {
        my $value = $::FORM{'target_milestone'};
        if (!defined $value || $value eq $::dontchange) {
            $value = $oldhash{'target_milestone'};
        }
        SendSQL("SELECT defaultmilestone FROM plans WHERE plan = " .
                SqlQuote($oldhash{'plan'}));
        if ($value eq FetchOneColumn()) {
            SendSQL("UNLOCK TABLES");
            PuntTryAgain("You must determine a target milestone for feature $id " .
                         "if you are going to accept it.  (Part of " .
                         "accepting a feature is giving an estimate of when it " .
                         "will be fixed.)");
        }
    }   
    if (defined $::FORM{'delta_ts'} && $::FORM{'delta_ts'} ne $delta_ts) {
        print "
<H1>Mid-air collision detected!</H1>
Someone else has made changes to this feature at the same time you were trying to.
The changes made were:
<p>
";
        DumpBugActivity($id, $delta_ts);
        my $longdesc = GetLongDescriptionAsHTML($id);
        my $longchanged = 0;

        if (length($longdesc) > $::FORM{'longdesclength'}) {
            $longchanged = 1;
            print "<P>Added text to the long description:<blockquote>";
            print substr($longdesc, $::FORM{'longdesclength'});
            print "</blockquote>\n";
        }
        SendSQL("unlock tables");
        print "You have the following choices: <ul>\n";
        $::FORM{'delta_ts'} = $delta_ts;
        print "<li><form method=post>";
        foreach my $i (keys %::FORM) {
            # Make sure we don't include the username/password fields in the
            # HTML.  If cookies are off, they'll have to reauthenticate after
            # hitting "submit changes anyway".
            # see http://bugzilla.mozilla.org/show_bug.cgi?id=15980
            if ($i !~ /^(Bugzilla|LDAP)_(login|password)$/) {
              my $value = value_quote($::FORM{$i});
              print qq{<input type=hidden name="$i" value="$value">\n};
            }
        }
        print qq{<input type=submit value="Submit my changes anyway">\n};
        print " This will cause all of the above changes to be overwritten";
        if ($longchanged) {
            print ", except for the changes to the description";
        }
        print qq{.</form>\n<li><a href="show_bug.cgi?id=$id">Throw away my changes, and go revisit feature $id</a></ul>\n};
        PutFooter();
        exit;
    }
        
    my %deps;
    if (defined $::FORM{'dependson'}) {
        my $me = "blocked";
        my $target = "dependson";
        for (1..2) {
            $deps{$target} = [];
            my %seen;
            foreach my $i (split('[\s,]+', $::FORM{$target})) {
                if ($i eq "") {
                    next;

                }
                SendSQL("select bug_id from bugs where bug_id = " .
                        SqlQuote($i));
                my $comp = FetchOneColumn();
                if ($comp ne $i) {
                    PuntTryAgain("$i is not a legal feature number");
                }
                if ($id eq $i) {
                    PuntTryAgain("You can't make a feature blocked or dependent on itself.");
                }
                if (!exists $seen{$i}) {
                    push(@{$deps{$target}}, $i);
                    $seen{$i} = 1;
                }
            }
            my @stack = @{$deps{$target}};
            while (@stack) {
                my $i = shift @stack;
                SendSQL("select $target from dependencies where $me = $i");
                while (MoreSQLData()) {
                    my $t = FetchOneColumn();
                    if ($t == $id) {
                        PuntTryAgain("Dependency loop detected!<P>" .
                                     "The change you are making to " .
                                     "dependencies has caused a circular " .
                                     "dependency chain.");
                    }
                    if (!exists $seen{$t}) {
                        push @stack, $t;
                        $seen{$t} = 1;
                    }
                }
            }

	    if ($me eq 'dependson') {
                my @deps   =  @{$deps{'dependson'}};
                my @blocks =  @{$deps{'blocked'}};
                my @union = ();
                my @isect = ();
                my %union = ();
                my %isect = ();
                foreach my $b (@deps, @blocks) { $union{$b}++ && $isect{$b}++ }
                @union = keys %union;
                @isect = keys %isect;
		if (@isect > 0) {
                    my $both;
                    foreach my $i (@isect) {
                       $both = $both . "#" . $i . " ";	
                    }
                    PuntTryAgain("Dependency loop detected!<P>" .
                                 "This feature can't be both blocked and dependent " .
                                 "on feature "  . $both . "!");
                }
            }
            my $tmp = $me;
            $me = $target;
            $target = $tmp;
        }
    }

    if (@::legal_keywords) {
        # There are three kinds of "keywordsaction": makeexact, add, delete.
        # For makeexact, we delete everything, and then add our things.
        # For add, we delete things we're adding (to make sure we don't
        # end up having them twice), and then we add them.
        # For delete, we just delete things on the list.
        my $changed = 0;
        if ($keywordaction eq "makeexact") {
            SendSQL("DELETE FROM keywords WHERE bug_id = $id");
            $changed = 1;
        }
        foreach my $keyword (@keywordlist) {
            if ($keywordaction ne "makeexact") {
                SendSQL("DELETE FROM keywords
                         WHERE bug_id = $id AND keywordid = $keyword");
                $changed = 1;
            }
            if ($keywordaction ne "delete") {
                SendSQL("INSERT INTO keywords 
                         (bug_id, keywordid) VALUES ($id, $keyword)");
                $changed = 1;
            }
        }
        if ($changed) {
            SendSQL("SELECT keyworddefs.name 
                     FROM keyworddefs, keywords
                     WHERE keywords.bug_id = $id
                         AND keyworddefs.id = keywords.keywordid
                     ORDER BY keyworddefs.name");
            my @list;
            while (MoreSQLData()) {
                push(@list, FetchOneColumn());
            }
            SendSQL("UPDATE bugs SET keywords = " .
                    SqlQuote(join(', ', @list)) .
                    " WHERE bug_id = $id");
        }
    }

    my $query = "$basequery\nwhere bug_id = $id";
    
# print "<PRE>$query</PRE>\n";

    if ($::comma ne "") {
        SendSQL($query);
        SendSQL("select delta_ts from bugs where bug_id = $id");
    } else {
        SendSQL("select now()");
    }
    $timestamp = FetchOneColumn();
    
    if (defined $::FORM{'comment'}) {
        AppendComment($id, $::FORM{'who'}, $::FORM{'comment'});
    }
    
    my $removedCcString = "";
    if (defined $::FORM{newcc} || defined $::FORM{removecc} || defined $::FORM{masscc}) {
        # Get the current CC list for this bug
        my %oncc;
        SendSQL("SELECT who FROM cc WHERE bug_id = $id");
        while (MoreSQLData()) {
            $oncc{FetchOneColumn()} = 1;
        }

        my (@added, @removed) = ();
        foreach my $pid (keys %cc_add) {
            # If this person isn't already on the cc list, add them
            if (! $oncc{$pid}) {
                SendSQL("INSERT INTO cc (bug_id, who) VALUES ($id, $pid)");
                push (@added, $cc_add{$pid});
                $oncc{$pid} = 1;
            }
        }
        foreach my $pid (keys %cc_remove) {
            # If the person is on the cc list, remove them
            if ($oncc{$pid}) {
                SendSQL("DELETE FROM cc WHERE bug_id = $id AND who = $pid");
                push (@removed, $cc_remove{$pid});
                $oncc{$pid} = 0;
            }
        }
        # Save off the removedCcString so it can be fed to processmail
        $removedCcString = join (",", @removed);

        # If any changes were found, record it in the activity log
        if (scalar(@removed) || scalar(@added)) {
            my $removed = join(", ", @removed);
            my $added = join(", ", @added);
            LogActivityEntry($id,"cc",$removed,$added);
        }
    }

    my %bucket;  # index is "$pool~$stage"
    SendSQL("SELECT pool,stage,effort FROM buckets WHERE id = $id");
    while(my($pool,$stage,$effort) = FetchSQLData()) {
      $bucket{"$pool~$stage"} = $effort;
    }
    for my $key (keys(%::FORM)) {
       next if($key !~ /^(.*?)-bUcK-(.*?)$/);
       my($pool,$stage) = ($1,$2);
       $pool = url_decode(url_decode($pool));
       $stage = url_decode(url_decode($stage));
       my $effort = $::FORM{$key};
       $effort = "" if !defined($effort);
       $bucket{"$pool~$stage"} ||= "";  # so it is defined
       next if($effort eq $bucket{"$pool~$stage"});
       if($effort =~ /^\s*-?[\d\.]+\s*$/) {
          SendSQL("REPLACE INTO buckets SET effort=$effort, pool=" . 
                  SqlQuote($pool) . ", stage=" .  SqlQuote($stage) . ", id=$id" );
       } elsif($effort eq "") {
          SendSQL("DELETE FROM buckets WHERE pool=" . 
                  SqlQuote($pool) . " AND stage=" .  SqlQuote($stage) . " AND id=$id" );
       };
       LogActivityEntry($id,"$stage $pool",$bucket{"$pool~$stage"},$effort);
    }

    if (defined $::FORM{'dependson'}) {
        my $me = "blocked";
        my $target = "dependson";
        for (1..2) {
            SendSQL("select $target from dependencies where $me = $id order by $target");
            my %snapshot;
            my @oldlist;
            while (MoreSQLData()) {
                push(@oldlist, FetchOneColumn());
            }
            my @newlist = sort {$a <=> $b} @{$deps{$target}};
            @dependencychanged{@oldlist} = 1;
            @dependencychanged{@newlist} = 1;

            while (0 < @oldlist || 0 < @newlist) {
                if (@oldlist == 0 || (@newlist > 0 &&
                                      $oldlist[0] > $newlist[0])) {
                    $snapshot{$newlist[0]} = SnapShotDeps($newlist[0], $me,
                                                          $target);
                    shift @newlist;
                } elsif (@newlist == 0 || (@oldlist > 0 &&
                                           $newlist[0] > $oldlist[0])) {
                    $snapshot{$oldlist[0]} = SnapShotDeps($oldlist[0], $me,
                                                          $target);
                    shift @oldlist;
                } else {
                    if ($oldlist[0] != $newlist[0]) {
                        die "Error in list comparing code";
                    }
                    shift @oldlist;
                    shift @newlist;
                }
            }
            my @keys = keys(%snapshot);
            if (@keys) {
                my $oldsnap = SnapShotDeps($id, $target, $me);
                SendSQL("delete from dependencies where $me = $id");
                foreach my $i (@{$deps{$target}}) {
                    SendSQL("insert into dependencies ($me, $target) values ($id, $i)");
                }
                foreach my $k (@keys) {
                    LogDependencyActivity($k, $snapshot{$k}, $me, $target);
                }
                LogDependencyActivity($id, $oldsnap, $target, $me);
            }

            my $tmp = $me;
            $me = $target;
            $target = $tmp;
        }
    }

    # When a feature changes plans and the old or new plan is associated
    # with a feature group, it may be necessary to remove the feature from the old
    # group or add it to the new one.  There are a very specific series of
    # conditions under which these activities take place, more information
    # about which can be found in comments within the conditionals below.
    if ( 
      # the "usebuggroups" parameter is on, indicating that plans
      # are associated with groups of the same name;
      Param('usebuggroups')

      # the user has changed the plan to which the feature belongs;
      && defined $::FORM{'plan'} 
        && $::FORM{'plan'} ne $::dontchange 
          && $::FORM{'plan'} ne $oldhash{'plan'} 
    ) {
        if (
          # the user wants to add the feature to the new plan's group;
          ($::FORM{'addtonewgroup'} eq 'yes' 
            || ($::FORM{'addtonewgroup'} eq 'yesifinold' 
                  && GroupNameToBit($oldhash{'plan'}) & $oldhash{'groupset'})) 

          # the new plan is associated with a group;
          && GroupExists($::FORM{'plan'})

          # the feature is not already in the group; (This can happen when the user
          # goes to the "edit multiple bugs" form with a list of bugs at least
          # one of which is in the new group.  In this situation, the user can
          # simultaneously change the bugs to a new plan and move the bugs
          # into that plan's group, which happens earlier in this script
          # and thus is already done.  If we didn't check for this, then this
          # situation would cause us to add the feature to the group twice, which
          # would result in the feature being added to a totally different group.)
          && !BugInGroup($id, $::FORM{'plan'})

          # the user is a member of the associated group, indicating they
          # are authorized to add bugs to that group, *or* the "usebuggroupsentry"
          # parameter is off, indicating that users can add bugs to a plan 
          # regardless of whether or not they belong to its associated group;
          && (UserInGroup($::FORM{'plan'}) || !Param('usebuggroupsentry'))

          # the associated group is active, indicating it can accept new bugs;
          && GroupIsActive(GroupNameToBit($::FORM{'plan'}))
        ) { 
            # Add the feature to the group associated with its new plan.
            my $groupbit = GroupNameToBit($::FORM{'plan'});
            SendSQL("UPDATE bugs SET groupset = groupset + $groupbit WHERE bug_id = $id");
        }

        if ( 
          # the old plan is associated with a group;
          GroupExists($oldhash{'plan'})

          # the feature is a member of that group;
          && BugInGroup($id, $oldhash{'plan'}) 
        ) { 
            # Remove the feature from the group associated with its old plan.
            my $groupbit = GroupNameToBit($oldhash{'plan'});
            SendSQL("UPDATE bugs SET groupset = groupset - $groupbit WHERE bug_id = $id");
        }

        print qq|</p>|;
    }
  
    # get a snapshot of the newly set values out of the database, 
    # and then generate any necessary feature activity entries by seeing 
    # what has changed since before we wrote out the new values.
    #
    my @newvalues = SnapShotBug($id);

    # for passing to processmail to ensure that when someone is removed
    # from one of these fields, they get notified of that fact (if desired)
    #
    my $origOwner = "";
    my $origQaContact = "";

    foreach my $c (@::log_columns) {
        my $col = $c;           # We modify it, don't want to modify array
                                # values in place.
        my $old = shift @oldvalues;
        my $new = shift @newvalues;
        if (!defined $old) {
            $old = "";
        }
        if (!defined $new) {
            $new = "";
        }
        if ($old ne $new) {

            # save off the old value for passing to processmail so the old
            # owner can be notified
            #
            if ($col eq 'assigned_to') {
                $old = ($old) ? DBID_to_name($old) : "";
                $new = ($new) ? DBID_to_name($new) : "";
                $origOwner = $old;
            }

            # ditto for the old qa contact
            #
            if ($col eq 'qa_contact') {
                $old = ($old) ? DBID_to_name($old) : "";
                $new = ($new) ? DBID_to_name($new) : "";
                $origQaContact = $old;
            }

            # If this is the keyword field, only record the changes, not everything.
            if ($col eq 'keywords') {
                ($old, $new) = DiffStrings($old, $new);
            }

            if ($col eq 'plan') {
                RemoveVotes($id, 0,
                            "This feature has been moved to a different plan");
            }
            LogActivityEntry($id,$col,$old,$new);
        }
    }
    
    print "<TABLE BORDER=1><TD><H2>Changes to feature $id submitted</H2>\n";
    SendSQL("unlock tables");

    my @ARGLIST = ();
    if ( $removedCcString ne "" ) {
        push @ARGLIST, ("-forcecc", $removedCcString);
    }
    if ( $origOwner ne "" ) {
        push @ARGLIST, ("-forceowner", $origOwner);
    }
    if ( $origQaContact ne "") { 
        push @ARGLIST, ( "-forceqacontact", $origQaContact);
    }
    push @ARGLIST, ($id, $::FORM{'who'});
    system ("./processmail",@ARGLIST);

    print "<TD><A HREF=\"show_bug.cgi?id=$id\">Back To Feature# $id</A></TABLE>\n";

    if ($duplicate) {
        # Check to see if Reporter of this feature is reporter of Dupe 
        SendSQL("SELECT reporter FROM bugs WHERE bug_id = " . SqlQuote($::FORM{'id'}));
        my $reporter = FetchOneColumn();
        SendSQL("SELECT reporter FROM bugs WHERE bug_id = " . SqlQuote($duplicate) . " and reporter = $reporter");
        my $isreporter = FetchOneColumn();
        SendSQL("SELECT who FROM cc WHERE bug_id = " . SqlQuote($duplicate) . " and who = $reporter");
        my $isoncc = FetchOneColumn();
        unless ($isreporter || $isoncc || ! $::FORM{'confirm_add_duplicate'}) {
            # The reporter is oblivious to the existance of the new feature and is permitted access
            # ... add 'em to the cc (and record activity)
            LogActivityEntry($duplicate,"cc","",DBID_to_name($reporter));
            SendSQL("INSERT INTO cc (who, bug_id) VALUES ($reporter, " . SqlQuote($duplicate) . ")");
        }
        AppendComment($duplicate, $::FORM{'who'}, "*** Feature $::FORM{'id'} has been marked as a duplicate of this bug. ***");
        if ( Param('strictvaluechecks') ) {
          CheckFormFieldDefined(\%::FORM,'comment');
        }
        SendSQL("INSERT INTO duplicates VALUES ($duplicate, $::FORM{'id'})");
        print "<TABLE BORDER=1><TD><H2>Duplicate notation added to feature $duplicate</H2>\n";
        system("./processmail", $duplicate, $::FORM{'who'});
        print "<TD><A HREF=\"show_bug.cgi?id=$duplicate\">Go To Feature# $duplicate</A></TABLE>\n";
    }

    foreach my $k (keys(%dependencychanged)) {
        print "<TABLE BORDER=1><TD><H2>Checking for dependency changes on feature $k</H2>\n";
        system("./processmail", $k, $::FORM{'who'});
        print "<TD><A HREF=\"show_bug.cgi?id=$k\">Go To Feature# $k</A></TABLE>\n";
    }

}

if (defined $::next_bug) {
    print("<P>The next feature in your list is:\n");
    $::FORM{'id'} = $::next_bug;
    print "<HR>\n";

    navigation_header();
    do "bug_form.pl";
} else {
    navigation_header();
    PutFooter();
}
