/* Copyright (c) Rick Dean 1997 */
/* <mishawaka@fdd.com> "Rick Dean" */

#include <assert.h>
#include <windows.h>
#include <commctrl.h>
#include "mishawaka.h"
#include "registry.h"
#include "status.h"
#include "inbound.h"
#include "dns.h"
#include "schedule.h"
#include "timer.h"
#include "debug.h"
#include "pop3.h"

static const char szAppName[] = "MishawakaApp"; // The name of this application
HWND hWndList;
HINSTANCE hInst;
HWND hMainWnd;

//  These globals are only changed by this file.
//  Other individual routines will declare them extern
struct timeval networkTimeout = {60,0}; /* measured in {seconds,milliseconds} */
char *spoolDirectoryName = "c:\\msdev\\projects\\mishawaka\\spool";
char *myHostnamePtr;
char *manualDNSServerPtr;
int autoMyHostname = 1;
int maxActiveInbound = 1;
char *hostsToConnectSMTPPtr;           // wildcards permitted
char *eAddressesToAcceptFromAnyonePtr; // wildcards permitted
char *forwardAnythingHostListPtr;      // wildcards permitted
char *keepEAddressList;                // wildcards permitted
int minutesToWaitBeforeResending = 120;
int mintuesToWaitBeforeBouncing = 60*24*3;
int returnBouncedMailToSender = 1;
struct timeval timeoutUntilDNSResend = {7,0};
int numDNSRetries = 4;
int dnsAquizition = IDC_MANUAL_DNS; 
int pauseNewSending = 0;
char *hostsToConnectPOP3Ptr;
int maxActivePOP3 = 1;
int warnIfConnectedDuringQuit = 1;
int warnIfWaitingDuringQuit = 1;
char *masterPOP3passwordPtr;
int useOSAccountsForPOP3 = 0;
char *pop3DomainNamePtr;

static void ReadRegistryPreferences(void)
{
    // Startup dialog box
    GetOurRegistryInt("warnIfConnectedDuringQuit",&warnIfConnectedDuringQuit);
    GetOurRegistryInt("warnIfWaitingDuringQuit",&warnIfWaitingDuringQuit);
    // DNS dialog box
    GetOurRegistryStr("myHostname",&myHostnamePtr,"myHostname");
    GetOurRegistryInt("dnsAquizition",&dnsAquizition);
    GetOurRegistryStr("manualDNSServers",&manualDNSServerPtr,"206.141.251.2     206.141.192.243");
    // Connect dialog box
    GetOurRegistryStr("hostsToConnectSMTP",&hostsToConnectSMTPPtr,"[127.0.0.1]    fdd.com    *.fdd.com");
    GetOurRegistryStr("hostsToConnectPOP3",&hostsToConnectPOP3Ptr,"[127.0.0.1]");
    GetOurRegistryInt("maxActiveInbound",&maxActiveInbound);
    // Route dialog box
    GetOurRegistryStr("eAddressesToAcceptFromAnyonePtr",&eAddressesToAcceptFromAnyonePtr,"<*@fdd.com>     <*@*.fdd.com>");
    GetOurRegistryStr("keepEAddressList",&keepEAddressList,"<*@localhost>    <*@*.fdd.com>    <*@fdd.com>");
    GetOurRegistryStr("forwardAnythingHostList",&forwardAnythingHostListPtr,"*.fdd.com    [127.0.0.1]");
    GetOurRegistryInt("returnBouncedMailToSender",&returnBouncedMailToSender);
    // Timeout dialog box
    GetOurRegistryInt("networkTimeout",&networkTimeout.tv_sec);
    GetOurRegistryInt("mintuesToWaitBeforeBouncing",&mintuesToWaitBeforeBouncing);
    GetOurRegistryInt("timeoutUntilDNSResend",&timeoutUntilDNSResend.tv_sec);
    GetOurRegistryInt("minutesToWaitBeforeResending",&minutesToWaitBeforeResending);
    GetOurRegistryInt("numDNSRetries",&numDNSRetries);
    // POP3 dialog box
    GetOurRegistryStr("masterPOP3password",&masterPOP3passwordPtr,"");
    GetOurRegistryInt("useOSAccountsForPOP3",&useOSAccountsForPOP3);
    GetOurRegistryStr("pop3DomainName",&pop3DomainNamePtr,"flam");
    // Menus
    GetOurRegistryInt("pauseNewSending",&pauseNewSending);
}

// Displays a pop-up message box. 
void __cdecl Msg( LPSTR formatStr, ... )
{
    char outputStr[1024];

    wvsprintf(outputStr, formatStr, (char *)(&formatStr+1));
    lstrcat(outputStr, "\r\n");
    MessageBox(NULL,outputStr,szAppName,MB_OK|MB_TOPMOST);
    assert(strlen(outputStr) < sizeof(outputStr));
}

static void UpdatePreferenceString(HWND hWndDlg,int itemNum,char **preferenceHdl)
{
    int stringLengthPlus1;

    if(*preferenceHdl != NULL)
        GlobalFree(*preferenceHdl);
    stringLengthPlus1 = GetWindowTextLength(GetDlgItem(hWndDlg,itemNum)) +1;
    *preferenceHdl = (char*) GlobalAlloc(GMEM_FIXED,stringLengthPlus1);
    if(*preferenceHdl != NULL) // if not out of memory
        GetDlgItemText(hWndDlg,itemNum,*preferenceHdl,stringLengthPlus1);
}

static BOOL CALLBACK AboutDialogProc(HWND hWndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)	
{
    int command;
    
    switch (uMsg) {
    case WM_INITDIALOG:
        return 1; // return "did not set focus"
    case WM_COMMAND:
        command = LOWORD(wParam);
        if(command == IDC_OK_NEXT)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,IDM_CREDITS,0);
        if(command == IDOK || command == IDC_OK_NEXT || 
            command == IDC_OK_PREV || command == IDCANCEL) {
            EndDialog(hWndDlg,0);
            return TRUE; 
        };
    }
    return FALSE;  // return "message not handled"
}

static BOOL CALLBACK CreditsDialogProc(HWND hWndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    int command;
    
    switch (uMsg) {
    case WM_INITDIALOG:
        return 1; // return "did not set focus"
    case WM_COMMAND:
        command = LOWORD(wParam);
        if(command == IDC_OK_NEXT)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,IDM_STARTUP,0);
        if(command == IDC_OK_PREV)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,IDM_ABOUT,0);
        if(command == IDOK || command == IDC_OK_NEXT || 
            command == IDC_OK_PREV || command == IDCANCEL) {
            EndDialog(hWndDlg,0);
            return TRUE; 
        };
    }
    return FALSE;  // return "message not handled"
}

static BOOL CALLBACK StartUpDialogProc(HWND hWndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)	
{
    int command;
    
    switch (uMsg) {
    case WM_INITDIALOG:
            CheckDlgButton(hWndDlg,IDC_WARN_OPEN_CONNECTIONS,warnIfConnectedDuringQuit);
            CheckDlgButton(hWndDlg,IDC_WARN_WAITING_MAIL,warnIfWaitingDuringQuit);
        return 1; // return "did not set focus"
    case WM_COMMAND:
        command = LOWORD(wParam);
//        if(command == IDM_ADD_SHORTCUT) {
//        };
        if(command == IDM_EXIT || command == IDM_HIDE)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,wParam,lParam); 
        if(command == IDC_OK_NEXT)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,IDM_DNS,0);
        if(command == IDC_OK_PREV)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,IDM_CREDITS,0);
        if(command == IDOK || command == IDC_OK_NEXT || command == IDC_OK_PREV) {
            warnIfConnectedDuringQuit = IsDlgButtonChecked(hWndDlg,IDC_WARN_OPEN_CONNECTIONS);
            warnIfWaitingDuringQuit = IsDlgButtonChecked(hWndDlg,IDC_WARN_WAITING_MAIL);
            SetOurRegistryInt("warnIfConnectedDuringQuit",warnIfConnectedDuringQuit);
            SetOurRegistryInt("warnIfWaitingDuringQuit",warnIfWaitingDuringQuit);
            EndDialog(hWndDlg,0);
            return TRUE; 
        };
        if(command == IDCANCEL || command == IDM_EXIT || command == IDM_HIDE) {
            EndDialog(hWndDlg,0);
            return TRUE; 
        };
    }
    return FALSE;  // return "message not handled"
}

static BOOL CALLBACK DNSDialogProc(HWND hWndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    int command;
    
    switch (uMsg) {
    case WM_INITDIALOG:
        CheckDlgButton(hWndDlg,IDC_AUTO_MY_HOSTNAME,autoMyHostname);
        SetDlgItemText(hWndDlg,IDC_MY_HOSTNAME,myHostnamePtr);
        SetDlgItemText(hWndDlg,IDE_MANUAL_DNS_SERVERS,manualDNSServerPtr);
        CheckDlgButton(hWndDlg,dnsAquizition,1);
        return 1; // return "did not set focus"
    case WM_COMMAND:
        command = LOWORD(wParam);
        if(command == IDC_OK_NEXT)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,IDM_CONNECT,0);
        if(command == IDC_OK_PREV)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,IDM_STARTUP,0);
        if(command == IDOK || command == IDC_OK_NEXT || command == IDC_OK_PREV) {
            UpdatePreferenceString(hWndDlg,IDC_MY_HOSTNAME,&myHostnamePtr);
            UpdatePreferenceString(hWndDlg,IDE_MANUAL_DNS_SERVERS,&manualDNSServerPtr);
            dnsAquizition = IsDlgButtonChecked(hWndDlg,IDC_MANUAL_DNS)*IDC_MANUAL_DNS +
                            IsDlgButtonChecked(hWndDlg,IDC_REVERSE_DNS)*IDC_REVERSE_DNS +
                            IsDlgButtonChecked(hWndDlg,IDC_FISH_REGISTRY)*IDC_FISH_REGISTRY +
                            IsDlgButtonChecked(hWndDlg,IDC_PROBE_SUBNET)*IDC_PROBE_SUBNET;
            autoMyHostname = IsDlgButtonChecked(hWndDlg,IDC_AUTO_MY_HOSTNAME);
            SetOurRegistryStr("myHostname",myHostnamePtr);
            SetOurRegistryStr("manualDNSServers",manualDNSServerPtr);
            SetOurRegistryInt("dnsAquizition",dnsAquizition);
            EndDialog(hWndDlg,0);
            return TRUE; 
        };
        if(command == IDCANCEL) { 
            EndDialog(hWndDlg,0);
            return TRUE; 
        };
    }
    return FALSE;  // return "message not handled"
}

static BOOL CALLBACK ConnectDialogProc(HWND hWndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    int command;
    
    switch (uMsg) {
    case WM_INITDIALOG:
        SetDlgItemText(hWndDlg,IDE_ACCEPT_SMTP_LIST,hostsToConnectSMTPPtr);
        SetDlgItemText(hWndDlg,IDE_ACCEPT_POP3_LIST,hostsToConnectPOP3Ptr);
        SetDlgItemInt(hWndDlg,IDE_MAX_INBOUND_CONNECTIONS,maxActiveInbound,FALSE);
        SetDlgItemInt(hWndDlg,IDE_MAX_OUTBOUND_CONNECTIONS,1,FALSE);
        SetDlgItemInt(hWndDlg,IDE_MAX_POP3_CONNECTIONS,maxActivePOP3,FALSE);
        return 1; // return "did not set focus"
    case WM_COMMAND:
        command = LOWORD(wParam);
        if(command == IDC_OK_NEXT)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,IDM_ROUTE,0);
        if(command == IDC_OK_PREV)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,IDM_DNS,0);
        if(command == IDOK || command == IDC_OK_NEXT || command == IDC_OK_PREV) {
            UpdatePreferenceString(hWndDlg,IDE_ACCEPT_SMTP_LIST,&hostsToConnectSMTPPtr);
            UpdatePreferenceString(hWndDlg,IDE_ACCEPT_POP3_LIST,&hostsToConnectPOP3Ptr);
            maxActiveInbound = GetDlgItemInt(hWndDlg,IDE_MAX_INBOUND_CONNECTIONS,NULL,FALSE);
            maxActivePOP3 = GetDlgItemInt(hWndDlg,IDE_MAX_POP3_CONNECTIONS,NULL,FALSE);
            SetOurRegistryStr("hostsToConnectSMTP",hostsToConnectSMTPPtr);
            SetOurRegistryStr("hostsToConnectPOP3",hostsToConnectPOP3Ptr);
            SetOurRegistryInt("maxActiveInbound",maxActiveInbound);
            SetOurRegistryInt("maxActivePOP3",maxActiveInbound);
            EndDialog(hWndDlg,0);
            return TRUE; 
        };
        if(command == IDCANCEL) {
            EndDialog(hWndDlg,0);
            return TRUE; 
        };
    }
    return FALSE;  // return "message not handled"
}

static BOOL CALLBACK RouteDialogProc(HWND hWndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    int command;
    
    switch (uMsg) {
    case WM_INITDIALOG:
        SetDlgItemText(hWndDlg,IDE_FORWARD_ANYTHING_HOSTS,forwardAnythingHostListPtr);
        SetDlgItemText(hWndDlg,IDC_ALWAYS_ACCEPT_EADDRESSES,eAddressesToAcceptFromAnyonePtr);
        SetDlgItemText(hWndDlg,IDE_KEEP_ADDRESSES,keepEAddressList);        
        return 1; // return "did not set focus"
    case WM_COMMAND:
        command = LOWORD(wParam);
        if(command == IDC_OK_NEXT)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,IDM_TIMEOUT,0);
        if(command == IDC_OK_PREV)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,IDM_CONNECT,0);
        if(command == IDOK || command == IDC_OK_NEXT || command == IDC_OK_PREV) {
            UpdatePreferenceString(hWndDlg,IDE_FORWARD_ANYTHING_HOSTS,&forwardAnythingHostListPtr);
            UpdatePreferenceString(hWndDlg,IDC_ALWAYS_ACCEPT_EADDRESSES,&eAddressesToAcceptFromAnyonePtr);
            UpdatePreferenceString(hWndDlg,IDE_KEEP_ADDRESSES,&keepEAddressList);
            SetOurRegistryStr("eAddressesToAcceptFromAnyonePtr",eAddressesToAcceptFromAnyonePtr);
            SetOurRegistryStr("keepEAddressList",keepEAddressList);
            SetOurRegistryStr("forwardAnythingHostList",forwardAnythingHostListPtr);
            EndDialog(hWndDlg,0);
            return TRUE; 
        };
        if(command == IDCANCEL) {
            EndDialog(hWndDlg,0);
            return TRUE; 
        };
    }
    return FALSE;  // return "message not handled"
}

static BOOL CALLBACK TimeoutDialogProc(HWND hWndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    int command;
    
    switch (uMsg) {
    case WM_INITDIALOG:
        SetDlgItemInt(hWndDlg,IDM_SMTP_TIMEOUT,networkTimeout.tv_sec,FALSE);
        SetDlgItemInt(hWndDlg,IDM_HOURS_UNTIL_BOUNCE,mintuesToWaitBeforeBouncing/60,FALSE);
        SetDlgItemInt(hWndDlg,IDM_SECONDS_UNTIL_DNS_RESEND,timeoutUntilDNSResend.tv_sec,FALSE);
        SetDlgItemInt(hWndDlg,IDM_NUM_DNS_RETRIES,numDNSRetries,FALSE);
        SetDlgItemInt(hWndDlg,IDE_MINUTES_UNTIL_RESEND,minutesToWaitBeforeResending,FALSE);
        CheckDlgButton(hWndDlg,IDM_DELIVER_BOUNCED_MAIL,returnBouncedMailToSender);
        return 1; // return "did not set focus"
    case WM_COMMAND:
        command = LOWORD(wParam);
        if(command == IDC_OK_NEXT)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,IDM_POP3,0);
        if(command == IDC_OK_PREV)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,IDM_ROUTE,0);
        if(command == IDOK || command == IDC_OK_NEXT || command == IDC_OK_PREV) {
            networkTimeout.tv_sec =        GetDlgItemInt(hWndDlg,IDM_SMTP_TIMEOUT,NULL,FALSE);
            mintuesToWaitBeforeBouncing =  GetDlgItemInt(hWndDlg,IDM_HOURS_UNTIL_BOUNCE,NULL,FALSE)*60;
            timeoutUntilDNSResend.tv_sec = GetDlgItemInt(hWndDlg,IDM_SECONDS_UNTIL_DNS_RESEND,NULL,FALSE);
            numDNSRetries =                GetDlgItemInt(hWndDlg,IDM_NUM_DNS_RETRIES,NULL,FALSE);
            minutesToWaitBeforeResending = GetDlgItemInt(hWndDlg,IDE_MINUTES_UNTIL_RESEND,NULL,FALSE);
            returnBouncedMailToSender    = IsDlgButtonChecked(hWndDlg,IDM_DELIVER_BOUNCED_MAIL);
            SetOurRegistryInt("returnBouncedMailToSender",returnBouncedMailToSender);
            SetOurRegistryInt("networkTimeout",networkTimeout.tv_sec);
            SetOurRegistryInt("mintuesToWaitBeforeBouncing",mintuesToWaitBeforeBouncing);
            SetOurRegistryInt("timeoutUntilDNSResend",timeoutUntilDNSResend.tv_sec);
            SetOurRegistryInt("numDNSRetries",numDNSRetries);
            SetOurRegistryInt("minutesToWaitBeforeResending",minutesToWaitBeforeResending);
            EndDialog(hWndDlg,0);
            DoScheduleTimer();
            return TRUE;
        } else if(command == IDCANCEL) {
            EndDialog(hWndDlg,0);
            return TRUE; 
        };
    }
    return FALSE;  // return "message not handled"
} 

static BOOL CALLBACK POP3DialogProc(HWND hWndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    int command;
    char changedPasswordStr[8];
    
    switch (uMsg) {
    case WM_INITDIALOG:
        if(masterPOP3passwordPtr[0] != '\0')
            SetDlgItemText(hWndDlg,IDE_MASTER_POP3_PASSWORD,"******");
        CheckDlgButton(hWndDlg,IDC_USE_OS_ACCOUNTS_FOR_POP3,useOSAccountsForPOP3);
        SetDlgItemText(hWndDlg,IDE_POP3_DOMAIN_NAME,pop3DomainNamePtr);        
        return 1; // return "did not set focus"
    case WM_COMMAND:
        command = LOWORD(wParam);
        if(command == IDC_OK_PREV)
            PostMessage(GetWindow(hWndDlg,GW_OWNER),WM_COMMAND,IDM_TIMEOUT,0);
        if(command == IDOK || command == IDC_OK_NEXT || command == IDC_OK_PREV) {
            GetDlgItemText(hWndDlg,IDE_MASTER_POP3_PASSWORD,changedPasswordStr,sizeof(changedPasswordStr));
            useOSAccountsForPOP3 = IsDlgButtonChecked(hWndDlg,IDC_USE_OS_ACCOUNTS_FOR_POP3);
            if(strcmp(changedPasswordStr,"******") != 0)
                UpdatePreferenceString(hWndDlg,IDE_MASTER_POP3_PASSWORD,&masterPOP3passwordPtr);
            UpdatePreferenceString(hWndDlg,IDE_POP3_DOMAIN_NAME,&pop3DomainNamePtr);
            SetOurRegistryStr("masterPOP3password",masterPOP3passwordPtr);
            SetOurRegistryStr("pop3DomainName",pop3DomainNamePtr);
            SetOurRegistryInt("useOSAccountsForPOP3",useOSAccountsForPOP3);
            EndDialog(hWndDlg,0);
            return TRUE;
        } else if(command == IDCANCEL) {
            EndDialog(hWndDlg,0);
            return TRUE; 
        };
    }
    return FALSE;  // return "message not handled"
} 

static void InitListViewWindow(HWND hWnd)
{
    RECT rectMainWindow;
  	LV_COLUMN lvC;      
    int index,columnWidthArr[] = { 140,100,100,60 };
    static char *columnNamesArr[] = {"name","from","to","status"};
	HICON hIcon;      
    HIMAGELIST hLarge,hSmall; 

    assert(hWndList == NULL);
    GetClientRect(hWnd,&rectMainWindow);
    InitCommonControls();
	hWndList = CreateWindowEx( 0L,WC_LISTVIEW,"",WS_VISIBLE | WS_CHILD | WS_BORDER | LVS_REPORT,
        0,0,rectMainWindow.right,rectMainWindow.bottom-statusHeight,hWnd,(HMENU)IDW_LISTVIEW,0,NULL);
    if(hWndList == NULL)
        return;
	lvC.mask = LVCF_FMT | LVCF_WIDTH | LVCF_TEXT | LVCF_SUBITEM;
	lvC.fmt = LVCFMT_LEFT;  // left align the column
    for (index = 0;index < sizeof(columnNamesArr)/sizeof(*columnNamesArr);index++) {
    	lvC.cchTextMax = 10;
        assert(sizeof(columnWidthArr)/sizeof(*columnWidthArr) > index);
        lvC.cx = columnWidthArr[index];            // width of the column, in pixels
    	lvC.pszText = columnNamesArr[index];
		lvC.iSubItem = index;
		if(ListView_InsertColumn(hWndList,index,&lvC) == -1)
			break;
	};
	hLarge = ImageList_Create(32,32,FALSE,1,0 );
	hSmall = ImageList_Create(16,16,FALSE,1,0 );
    hIcon = LoadIcon(hInst,MAKEINTRESOURCE(822));
    ImageList_AddIcon(hLarge, hIcon);
    ImageList_AddIcon(hSmall, hIcon);
	ListView_SetImageList(hWndList,hLarge,LVSIL_NORMAL);  
	ListView_SetImageList(hWndList,hSmall,LVSIL_SMALL);  
    //  How do I make the transparent part of icons be white not black???????????
    ResetListView(hWndList);  // schedule.c
}

static void DoChangeView(HWND hWndList,int command)
{
    static unsigned int viewStyleArr[] = {LVS_ICON,LVS_SMALLICON,LVS_LIST,LVS_REPORT};
    DWORD dwStyle;
  
    assert(hWndList);
    command -= IDM_LARGEICON;
    assert(command >= 0 && command < sizeof(viewStyleArr)/sizeof(*viewStyleArr));
    dwStyle = GetWindowLong(hWndList,GWL_STYLE);
    if ((dwStyle & LVS_TYPEMASK) != viewStyleArr[command])
        SetWindowLong(hWndList,GWL_STYLE,(dwStyle & ~LVS_TYPEMASK) | viewStyleArr[command]);
}

static void DoCommandOnSelectMessages(HWND hWndList,int command)
{
    int listIndex;
    LV_ITEM listViewItem;
    char itemNameArr[100];

    assert(hWndList);
    listViewItem.cchTextMax = sizeof(itemNameArr);
    listViewItem.pszText = itemNameArr;
    listViewItem.iSubItem = 0;
    listViewItem.mask = LVIF_TEXT | LVIF_STATE | LVIF_PARAM;
    listViewItem.stateMask = 0xFFFFFFFF;
    for(listIndex = ListView_GetItemCount(hWndList)-1;listIndex >= 0;listIndex--) {
        listViewItem.iItem = listIndex;
        if(!ListView_GetItem(hWndList,&listViewItem)) {
            Printf("  get item error\r\n");
            continue;
        };
        if((listViewItem.state & LVIS_SELECTED) == 0) // if not selected
            continue;
        DoCommandOnEnvleope(itemNameArr,listViewItem.lParam,command);  // schedule.c
    };  
}

static void ListViewPopUpMenu(HWND hWnd,POINT point,HINSTANCE hInst)
{
    HMENU hMenu;
    HMENU hMenuTrackPopup;

    hMenu = LoadMenu(hInst, "FileContextMenu");
    if(hMenu == NULL)
        return;
    hMenuTrackPopup = GetSubMenu(hMenu,0);
    TrackPopupMenu(hMenuTrackPopup, TPM_LEFTALIGN | TPM_RIGHTBUTTON,point.x,point.y,0,hWnd,NULL);
    DestroyMenu(hMenu);
}

//  window handle is for ListView window
static int NumItemsSelected(HWND hWndList)
{
    int nextItem,numSelected;

    assert(hWndList);
    for(nextItem=-1,numSelected=0;;numSelected++) {
        nextItem = ListView_GetNextItem(hWndList,nextItem,LVNI_SELECTED);
        if(nextItem == -1)
            break;
    };
    return numSelected;
}

static void DoNotify(HWND hWnd,UINT message,WPARAM wParam,LPARAM lParam)
{
	NM_LISTVIEW *pNm = (NM_LISTVIEW *)lParam;
	POINT point;	
#ifdef HIT_TEST_WORKS
	LV_HITTESTINFO lvH;
	int itemIndex;
#endif

	if(wParam != IDW_LISTVIEW)
		return;
	switch(pNm->hdr.code)	{
	case NM_RCLICK:		// right mouse button has been clicked
#ifdef HIT_TEST_WORKS
        lvH.pt = pNm->ptAction;
        itemIndex = ListView_HitTest(hWndList,&lvH);// doesn't work!!!!
        //Printf("missing arg = %d\r\n"); // will give correct value. What the fuck?!?!?
        Printf("numItemsSelected %d\r\n",NumItemsSelected(hWndList)); // will give correct value. What the fuck
        if(lvH.iItem != -1) {
#else
        if(NumItemsSelected(hWndList) > 0) {
#endif
			GetCursorPos(&point);
            ListViewPopUpMenu(hWnd,point,hInst);
		};
		break;
	};
	return;
}

static void UpdateMenus(HWND hWnd)
{   
    HMENU menuHdl;
    int numSelected;

    numSelected = NumItemsSelected(hWndList);
    menuHdl = GetMenu(hWnd);
    EnableMenuItem(menuHdl,IDM_VIEW_CONTENTS,numSelected>0&&numSelected<4?MF_ENABLED:MF_GRAYED);  
    EnableMenuItem(menuHdl,IDM_EDIT_ENVELOPE,numSelected==1?MF_ENABLED:MF_GRAYED);  
    EnableMenuItem(menuHdl,IDM_DELETE_ENVELOPE,numSelected>0?MF_ENABLED:MF_GRAYED);  
    EnableMenuItem(menuHdl,IDM_NEXT_ENVELOPE,numSelected>0?MF_ENABLED:MF_GRAYED);  
    EnableMenuItem(menuHdl,IDM_KEEPER_ENVELOPE,numSelected>0?MF_ENABLED:MF_GRAYED);  
    EnableMenuItem(menuHdl,IDM_USER_NOT_FOUND,numSelected>0?MF_ENABLED:MF_GRAYED);  
    EnableMenuItem(menuHdl,IDM_UNDELIVERABLE,numSelected>0?MF_ENABLED:MF_GRAYED);  
    EnableMenuItem(menuHdl,IDM_FUCK_YOURSELF,numSelected>0?MF_ENABLED:MF_GRAYED);  
    CheckMenuItem(menuHdl,IDM_DEBUG_WINDOW,IsDebugShowing()?MF_CHECKED:MF_UNCHECKED);
    CheckMenuItem(menuHdl,IDM_PAUSE_NEW,pauseNewSending?MF_CHECKED:MF_UNCHECKED);
    CloseHandle(menuHdl);
}

static void DoPauseNew(HWND hWnd)
{    
    pauseNewSending = !pauseNewSending;
    SetOurRegistryInt("pauseNewSending",pauseNewSending);
    if(!pauseNewSending)
        DoScheduleTimer();  // schedule.c
}

static void DoKeyDown(HWND hWnd,WPARAM wParam)
{
    static tickOfLastEscape;

    if(wParam == VK_ESCAPE)
        if(GetTickCount() - tickOfLastEscape < GetDoubleClickTime()) // in milliseconds
            ShowWindow(hWnd,SW_HIDE);
        else 
            tickOfLastEscape = GetTickCount();
}

static void DoSize(HWND hWnd,LPARAM lParam)
{
    if(lParam == 0)  // if size is 0 by 0 (being minimized)
        return;
    MoveStatusWindow(hWnd);  // status.h
    MoveWindow(hWndList,0,0,LOWORD(lParam),HIWORD(lParam)-statusHeight,TRUE);
}

static void DoShowWindow(HWND hWnd,LPARAM wParam)
{
    if(wParam) { // if becomming showing
        InitListViewWindow(hWnd);
    } else { // else becomming hidden
        assert(hWndList);
        DestroyWindow(hWndList);
        hWndList = NULL;
    };
}

static void DoCreate(HWND hWnd)
{
    InitStatusWindow(hWnd);  // status.c
}

//  Mishawaka specific event.
static void DoInit(HWND hWnd)
{
    int returnStatus; 
    WSADATA wsaData;

    hMainWnd = hWnd;
    if((returnStatus = WSAStartup(MAKEWORD(1,1), &wsaData)) != 0) {
       Msg("WSAStartup() failed (%d)\n",returnStatus);
       exit(1);
    };
    InitSchedule(hWnd);  // schedule.c
    InitInbound(hWnd);  // inbound.c
    InitPOP3(hWnd);     // pop3.c
    SetStatusText("");
}

//  This is before the window has been hidden.  (before DestroyWindow())
static void DoClose(HWND hWnd)
{
    extern volatile int numActiveInbound,numActivePOP3;

    if(warnIfConnectedDuringQuit && (numActiveInbound > 0 || IsAnyMail(SENDING) || numActivePOP3 > 0)  &&
        MessageBox(NULL,"Quit with connections open?\r\n",szAppName,MB_YESNO|MB_TOPMOST) == IDNO)
        return;
    if(warnIfWaitingDuringQuit && IsAnyMail(WAITING) &&
        MessageBox(NULL,"Quit with mail waiting for delivery?\r\n",szAppName,MB_YESNO|MB_TOPMOST) == IDNO)
        return;
    DestroyWindow(hWnd);
}

// This is after the widow has been hidden, before it is destroyed.  (after DestroyWindow())
static void DoQuit(HWND hWnd)
{
    RECT windowRect;

    QuitSchedule(hWnd);
    GetWindowRect(hWnd,&windowRect);
    if(!IsIconic(hWnd)) {  // if window non iconified or hidden
        SetOurRegistryInt("left",windowRect.left);
        SetOurRegistryInt("top",windowRect.top);
        SetOurRegistryInt("width",windowRect.right-windowRect.left);
        SetOurRegistryInt("height",windowRect.bottom-windowRect.top);
    };
    //WSACleanup();  // why bother?  keep program size small 
}

static int DoMenuCommand(HWND hWnd,UINT message,WPARAM wParam,LPARAM lParam)
{
    int identifier;

    identifier = LOWORD(wParam);
    switch(identifier) {
	case IDM_TEST1:
        InitDNS(); // dns.c
       	break;
    case IDM_HIDE:
        ShowWindow(hWnd,SW_HIDE);
        break;
    case IDM_EXIT:
        SendMessage(hWnd,WM_CLOSE,0,0);
        break;
    case IDM_ABOUT:
        DialogBox(hInst,MAKEINTRESOURCE(IDD_ABOUT),hWnd,(DLGPROC)AboutDialogProc);
        break;
    case IDM_CREDITS:
        DialogBox(hInst,MAKEINTRESOURCE(IDD_CREDITS),hWnd,(DLGPROC)CreditsDialogProc);
        break;
    case IDM_STARTUP:
        DialogBox(hInst,MAKEINTRESOURCE(IDD_STARTUP),hWnd,(DLGPROC)StartUpDialogProc);
        break;
    case IDM_DNS:
        DialogBox(hInst,MAKEINTRESOURCE(IDD_DNS),hWnd,(DLGPROC)DNSDialogProc);
        break;
    case IDM_CONNECT:
        DialogBox(hInst,MAKEINTRESOURCE(IDD_CONNECT),hWnd,ConnectDialogProc);
        break;
    case IDM_ROUTE:
        DialogBox(hInst,MAKEINTRESOURCE(IDD_ROUTE),hWnd,RouteDialogProc);
        break;
    case IDM_TIMEOUT:
        DialogBox(hInst,MAKEINTRESOURCE(IDD_TIMEOUT),hWnd,(DLGPROC)TimeoutDialogProc);
        break;
    case IDM_POP3:
        DialogBox(hInst,MAKEINTRESOURCE(IDD_POP3),hWnd,(DLGPROC)POP3DialogProc);
        break;
    case IDM_DELETE_ENVELOPE: case IDM_USER_NOT_FOUND: case IDM_UNDELIVERABLE: case IDM_FUCK_YOURSELF:
    case IDM_EDIT_ENVELOPE: case IDM_VIEW_CONTENTS: case IDM_NEXT_ENVELOPE: case IDM_KEEPER_ENVELOPE:
        DoCommandOnSelectMessages(hWndList,identifier);
        break;
    case IDM_PAUSE_NEW:
        DoPauseNew(hWnd);
        ResetListView(hWndList);  // schedule.c
        break;   
    case IDM_LARGEICON: case IDM_SMALLICON: case IDM_LISTVIEW: case IDM_REPORTVIEW:
        DoChangeView(hWndList,identifier);
        break;
    case IDM_REFRESH_VIEW:
        ResetListView(hWndList);  // schedule.c
        break;
    case IDM_DEBUG_WINDOW:
        ToggleDebugShow();  // debug.c
        break;
    default:  // fall through from above
	    return DefWindowProc(hWnd, message, wParam, lParam);
    };
}

LRESULT CALLBACK MainWindowProcedure(HWND hWnd,UINT message,WPARAM wParam,LPARAM lParam)
{
    switch(message) { 
	case WM_COMMAND:
        return DoMenuCommand(hWnd,message,wParam,lParam);
    case MISH_INBOUND_MAIL_ACCEPT: /* Notification if a socket connection is pending. (user defined event) */
        DoInboundMailAccept(hWnd,(SOCKET) wParam,lParam);  // inbound.c 
        break;
    case MISH_INBOUND_POP3_ACCEPT: /* Notification if a socket connection is pending. (user defined event) */
        DoInboundPOP3Accept(hWnd,(SOCKET) wParam,lParam);  // inbound.c 
        break;
    case MISH_MAIL_RECEIVED: // inbound.c has successfully received a letter 
        DoRouteInboundMail(hWnd,(HANDLE)wParam);  // schedule.c 
        break;
    case MISH_MAIL_STATUS: // outbound.c has finished a send attempt 
        DoMailStatus(hWnd,(Envelope *)wParam);  // schedule.c 
        break;
	case WM_NOTIFY:
        DoNotify(hWnd,message,wParam,lParam);
		break;
    case WM_KEYDOWN:
        DoKeyDown(hWnd,wParam);
        break;
    case WM_SIZE: 
        DoSize(hWnd,lParam);
        break;
    case WM_SHOWWINDOW: 
        DoShowWindow(hWnd,wParam);
        break;
    case WM_CREATE:
        DoCreate(hWnd);
        break;
    case MISH_INIT:
        DoInit(hWnd);
        break;
	case WM_CLOSE:
        DoClose(hWnd);
		break;
	case WM_DESTROY:
        DoQuit(hWnd);
        PostQuitMessage(0);
		break;
    case WM_INITMENU:
        UpdateMenus(hWnd);
        break;
	default: 
		return DefWindowProc(hWnd,message,wParam,lParam);
	}
	return (0);
}

static void InitInstance(HINSTANCE hInstance,LPSTR lpCmdLine,int nCmdShow)
{
	HWND hWnd;
    int top=30,left=30,width=600,height=400;
    int screenWidth,screenHeight,firstTimeFlag;
	
    screenWidth = GetSystemMetrics(SM_CXSCREEN);
    screenHeight = GetSystemMetrics(SM_CYSCREEN);
    top = -1000;
    GetOurRegistryInt("top",&top);
    firstTimeFlag = (top == -1000);  // if top didn't exist then this is the first time run.
    if(top < 0 || top > screenHeight)
        top = 20;
    GetOurRegistryInt("left",&left);
    if(left < 0 || left > screenWidth)
        left = 20;
    GetOurRegistryInt("width",&width);
    if(width < 20 || width > screenWidth)
        width = screenWidth/2;
    GetOurRegistryInt("height",&height);
    if(height < 20 || height > screenHeight)
        height = screenHeight/4;
	hWnd = CreateWindow(szAppName,"Mishawaka Internet Mail Server",WS_OVERLAPPEDWINDOW,left,top,
        width,height,NULL,NULL,hInstance,NULL);
	if(hWnd == NULL) {
        char errorStr[1024];
        FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM,NULL,GetLastError(),MAKELANGID(LANG_NEUTRAL,SUBLANG_DEFAULT),
            errorStr,1024,NULL);
        Msg("Could not create window\n(%s)",errorStr);
		exit(0);
	}
    if(strncmp(lpCmdLine,"/hide",5) == 0) {  // needs better pattern matcher
        ShowWindow(hWnd,SW_HIDE);
        ShowWindow(hWnd,SW_HIDE);
    } else {
        ShowWindow(hWnd,nCmdShow);
	    UpdateWindow(hWnd);
    };
    PostMessage(hWnd,MISH_INIT,0,0);
    if(firstTimeFlag) 
        PostMessage(hWnd,WM_COMMAND,IDM_ABOUT,0);
}

static void InitApplication(HINSTANCE hInstance,LPSTR lpCmdLine)
{
	WNDCLASS  wc;
    HWND hWnd;

    hWnd = FindWindow(szAppName, NULL);  // To see if another instance is running
    if (hWnd) { // if another copy of this program is running
        if(!IsWindowVisible(hWnd) || IsIconic(hWnd)) 
            ShowWindow(hWnd,SW_RESTORE);
        SetForegroundWindow(hWnd);
        exit(0);
    };
	// build window class data structure for main window.
	wc.style         = CS_HREDRAW | CS_VREDRAW;
	wc.lpfnWndProc   = (WNDPROC)MainWindowProcedure;
	wc.cbClsExtra    = 0;
	wc.cbWndExtra    = 0;
	wc.hInstance     = hInstance;
	wc.hIcon         = LoadIcon (hInstance, szAppName);
	wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = (HBRUSH)(COLOR_WINDOW+1);
    wc.lpszMenuName  = "mishawakamenu";  
	wc.lpszClassName = szAppName;
    if(!RegisterClass(&wc)) {
	   // MessageBox (NULL, "Could not register application class!", NULL, MB_OK );
		exit(0);
	};
}

int APIENTRY WinMain(HINSTANCE hInstance,HINSTANCE hPrevInstance,LPSTR lpCmdLine,int nCmdShow)
{
	MSG msg;
	HANDLE hAccelTable;

    hInst = hInstance;
    InitApplication(hInstance,lpCmdLine);
    InitDebugWindow();   
    Printf("Mishawaka "VERSION_STRING" built "__DATE__" "__TIME__"\r\n"); // compiler string concatenation
    ReadRegistryPreferences();
	InitInstance(hInstance,lpCmdLine,nCmdShow); 
	hAccelTable = LoadAccelerators (hInstance, szAppName);
    while(GetMessage(&msg, NULL, 0, 0)) {  /* main message loop */
        if (!TranslateAccelerator(msg.hwnd,hAccelTable,&msg)) {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        };
    };
    return (msg.wParam);
}


