/* Copyright (c) Rick Dean 1997 */
/* <mishawaka@fdd.com> Rick Dean */

#include <assert.h>
#define WINSOCK2_H_
#include <winsock2.h>
#include <setjmp.h>

#include "mishawaka.h"
#include "status.h"
#include "dns.h"
#include "outbound.h"
#include "timer.h"
#include "debug.h"

#define MAX_ERROR_STRING_LENGTH (1024)

// returns 1 on internal error (file open failure)
// or 0 otherwise.
int WriteToSocketFromFile(SOCKET sock,jmp_buf abortJmpBuf,char *filenameStr)
{
    extern struct timeval networkTimeout;
    struct fd_set myFD_set;
    char sendingLineArr[2048],*sendingLinePtr; // points to first empty char
    int numBytesSent,numBytesRead,totalBytesSent;
    HANDLE fileHdl = NULL;

    fileHdl = CreateFile(filenameStr,GENERIC_READ,FILE_SHARE_READ,NULL,OPEN_EXISTING,FILE_FLAG_SEQUENTIAL_SCAN,NULL);
    if(fileHdl == INVALID_HANDLE_VALUE) 
        longjmp(abortJmpBuf,INTERNAL_ERROR);
    SendingStarted(fileHdl,"xxx");  // timer.c
    sendingLinePtr = sendingLineArr;
    totalBytesSent = 0;
    for(;;) { // for every send attempt
        if(ReadFile(fileHdl,sendingLinePtr,sizeof(sendingLineArr)-(sendingLinePtr-sendingLineArr),
                    &numBytesRead,NULL) == FALSE && GetLastError() != ERROR_HANDLE_EOF) {
            CloseHandle(fileHdl);
            return 1; // return "internal error" (file open failure)
        };
        sendingLinePtr+=numBytesRead;
        if(sendingLinePtr == sendingLineArr)  // if buffer to send is empty
            break;  
        FD_ZERO(&myFD_set);
        FD_SET(sock,&myFD_set); // list our socket as the one to check
        if(select(sock,NULL,&myFD_set,NULL,&networkTimeout) == 0){//if sending timeout occurrs
            CloseHandle(fileHdl);
            longjmp(abortJmpBuf,NETWORK_PROBLEM);
        };
        numBytesSent = send(sock,sendingLineArr,sendingLinePtr-sendingLineArr,0);
        if(numBytesSent == SOCKET_ERROR || numBytesSent == 0) {
            CloseHandle(fileHdl);
            longjmp(abortJmpBuf,NETWORK_PROBLEM);
        }
        if(numBytesSent < sendingLinePtr-sendingLineArr)
            memmove(sendingLineArr,sendingLineArr+numBytesSent,sendingLinePtr-sendingLineArr-numBytesSent); // shift left
        sendingLinePtr -= numBytesSent;
        totalBytesSent += numBytesSent;
        SendingProgress(totalBytesSent);
    };
    CloseHandle(fileHdl); 
    return 0;  // return "no internal error"
}

// returns 0 = OK
// Note: Doesn't allow for whitespace at the beginning of the line.
int ReceiveExpected(SOCKET sock,char *expectedPtr,char receivedLineArr[],int maxReceivedLength,jmp_buf abortJmpBuf)
{
    extern struct timeval networkTimeout;
    struct fd_set myFD_set;
    char *cursorPtr; // points to first empty char
    int numBytesReceived,returnStatus;

    assert(maxReceivedLength > 10);
    cursorPtr = receivedLineArr;   
    for(;;) { // for all the reads until a newline or timeout is found
        FD_ZERO(&myFD_set);
        FD_SET(sock,&myFD_set);
        returnStatus = select(sock,&myFD_set,NULL,NULL,&networkTimeout); 
        if(returnStatus == 0) { // if receiving timeout occurrs
            Printf("##select() timeout\r\n");
            longjmp(abortJmpBuf,NETWORK_PROBLEM);   
        };
        if(returnStatus == SOCKET_ERROR) { // if receiving timeout occurrs
            Printf("##select() error %d\r\n",WSAGetLastError());
            longjmp(abortJmpBuf,NETWORK_PROBLEM);   
        };
//Printf("got select have=%d asking=%d\r\n",cursorPtr-receivedLineArr,maxReceivedLength-(cursorPtr-receivedLineArr)-1);
        numBytesReceived = recv(sock,cursorPtr,maxReceivedLength-(cursorPtr-receivedLineArr)-1,0);
        if(numBytesReceived == SOCKET_ERROR) {
            Printf("##recv() failed %d\r\n",WSAGetLastError());
            longjmp(abortJmpBuf,NETWORK_PROBLEM);
        };
        cursorPtr += numBytesReceived;
        *cursorPtr = '\0';
//Printf("received \"%s\"\r\n",receivedLineArr);
        if(strchr(receivedLineArr,'\n')) // if end of line found
            break;
        if(cursorPtr - receivedLineArr == maxReceivedLength-1) // if out of buffer memory
            cursorPtr -= 10;  // keep overwriting last 10 character
    };
    Printf("received: %s",receivedLineArr);
    if(_strnicmp(expectedPtr,receivedLineArr,strlen(expectedPtr)) == 0) // case insensitive
        return 0;
    *cursorPtr = '\0';
    return 1;
}

void __cdecl SocketPrintf(SOCKET sock,jmp_buf abortJmpBuf,LPSTR formatStr, ... )
{
    static char outputStr[1024];

    wvsprintf(outputStr,formatStr,(char *)(&formatStr+1));  
    Printf("sending: %s",outputStr);
    assert(strlen(outputStr) < sizeof(outputStr)-1);  // -1 for the '\0'
    if(send(sock,outputStr,strlen(outputStr),0) == SOCKET_ERROR)  // could block?
        longjmp(abortJmpBuf,NETWORK_PROBLEM);
}

void PrintfIPAddress(int ipAddress)
{
    Printf("%d.%d.%d.%d\r\n",(ipAddress&0xff000000)>>24,(ipAddress&0x00ff0000)>>16,
        (ipAddress&0x0000ff00)>>8,ipAddress&0xff);
}

static SendFileError SendSMTPWithoutClosing(SOCKET sock,Envelope * const envelopePtr,char *toHostPtr,
    char receivedLineArr[MAX_ERROR_STRING_LENGTH],MailExchanger mailExchangerArr[MAX_NUM_OF_MAIL_EXCHANGERS])
{
    SOCKADDR_IN sockaddr_in;
    int mailExchangerIndex,returnStatus;
    jmp_buf abortJmpBuf;
    extern char *myHostnamePtr;

    returnStatus = setjmp(abortJmpBuf);
    if(returnStatus != 0)
        return returnStatus;
    sockaddr_in.sin_family = AF_INET;  /* address family internet */
    sockaddr_in.sin_port = htons(25);// connect to SMTP mail port
    for(mailExchangerIndex=0;mailExchangerIndex < MAX_NUM_OF_MAIL_EXCHANGERS;mailExchangerIndex++) {
        Printf("attempting to connect to #%d priority=%d ipAddress=",mailExchangerIndex,mailExchangerArr[mailExchangerIndex].priority);
        PrintfIPAddress(mailExchangerArr[mailExchangerIndex].ipAddress);
        if(mailExchangerArr[mailExchangerIndex].ipAddress == 0)
           continue;  // go to next mail exchanger
        sockaddr_in.sin_addr.s_addr = htonl(mailExchangerArr[mailExchangerIndex].ipAddress); 
        if(connect(sock,(SOCKADDR*)&sockaddr_in,sizeof(SOCKADDR)) == SOCKET_ERROR) {
           returnStatus = WSAGetLastError();
           Printf("connect failed! WSAGetLastError()=%d\r\n",returnStatus);
           if(returnStatus == WSAECONNREFUSED || returnStatus == WSAENETUNREACH || 
               returnStatus == WSAETIMEDOUT || returnStatus == WSAEHOSTUNREACH)
               continue; // try next mail exchanger (Should we really recycle the socket? but if socket() failed, we would be hosed!)
           return NETWORK_PROBLEM;  // for all other errors, a different destination won't help
        };
        if(ReceiveExpected(sock,"2",receivedLineArr,MAX_ERROR_STRING_LENGTH,abortJmpBuf)) 
            return NETWORK_PROBLEM;
        if(strncmp(receivedLineArr+4,myHostnamePtr,strlen(myHostnamePtr)) == 0)
            return MAIL_LOOP;
        SocketPrintf(sock,abortJmpBuf,"HELO %s\r\n",toHostPtr);
        ReceiveExpected(sock,"2",receivedLineArr,MAX_ERROR_STRING_LENGTH,abortJmpBuf);//ignore response
        SocketPrintf(sock,abortJmpBuf,"MAIL From: %s\r\n",envelopePtr->fromAddressArr);
        if(ReceiveExpected(sock,"2",receivedLineArr,MAX_ERROR_STRING_LENGTH,abortJmpBuf))
            return SMTP_ERROR;
        SocketPrintf(sock,abortJmpBuf,"RCPT To: %s\r\n",envelopePtr->toAddressArr);
        if(ReceiveExpected(sock,"2",receivedLineArr,MAX_ERROR_STRING_LENGTH,abortJmpBuf))
            return SMTP_ERROR;
        SocketPrintf(sock,abortJmpBuf,"DATA\r\n");
        if(ReceiveExpected(sock,"354",receivedLineArr,MAX_ERROR_STRING_LENGTH,abortJmpBuf))
            return SMTP_ERROR;
        Printf("sending data\r\n");
        if(WriteToSocketFromFile(sock,abortJmpBuf,envelopePtr->eight22FilenameArr)) // if internal error (file open failure)
            longjmp(abortJmpBuf,INTERNAL_ERROR);
        if(ReceiveExpected(sock,"2",receivedLineArr,MAX_ERROR_STRING_LENGTH,abortJmpBuf))
            return SMTP_ERROR;
        SocketPrintf(sock,abortJmpBuf,"QUIT\r\n");  // ignore possible error
        return SUCCESS_FILE_SENT;
    };
    return NETWORK_PROBLEM;
}

// returns non-zero on error
static int ExtractDomainName(char *domainNameStr,char *emailAddressStr)
{
    char *atPtr,*greaterThanPtr;

    atPtr = strchr(emailAddressStr,'@');
    if(atPtr == NULL)
        return 1;
    greaterThanPtr = strchr(atPtr,'>');
    assert(greaterThanPtr != NULL);
    memcpy(domainNameStr,atPtr+1,greaterThanPtr-atPtr-1);
    domainNameStr[greaterThanPtr-atPtr-1] = '\0';
    return 0;
}

static void SendSMTP(Envelope *envelopePtr)
{
    MailExchanger mailExchangerArr[MAX_NUM_OF_MAIL_EXCHANGERS];
    SOCKET sock;
    char toHostArr[MAX_E_ADDRESS_LENGTH],receivedLineArr[MAX_ERROR_STRING_LENGTH];

    assert(strlen(envelopePtr->toAddressArr) < MAX_E_ADDRESS_LENGTH);
    assert(strlen(envelopePtr->fromAddressArr) < MAX_E_ADDRESS_LENGTH);
    assert(strlen(envelopePtr->eight22FilenameArr) < MAX_822FILENAME_LENGTH);
    ExtractDomainName(toHostArr,envelopePtr->toAddressArr);
    switch(GetMXforName(toHostArr,mailExchangerArr)) {
    case 1:
        envelopePtr->status = NETWORK_PROBLEM;
        return;
    case 2: case 3:
        envelopePtr->status = DESTINATION_DOES_NOT_EXIST;
        return;
    };
    sock = socket(AF_INET,SOCK_STREAM,0);
    if(sock == INVALID_SOCKET) {
        envelopePtr->status = NETWORK_PROBLEM;
        return;
    };
    envelopePtr->status = SendSMTPWithoutClosing(sock,envelopePtr,toHostArr,receivedLineArr,mailExchangerArr);
    if(envelopePtr->status == SMTP_ERROR) {
        envelopePtr->smtpErrorPtr = (char*) GlobalAlloc(GMEM_FIXED,strlen(receivedLineArr)+1);
        if(envelopePtr->smtpErrorPtr) 
            strcpy(envelopePtr->smtpErrorPtr,receivedLineArr);
    };
    closesocket(sock);
    return;
}

WINAPI OutboundThreadMain(Envelope *envelopePtr)
{
    extern HWND hMainWnd;

    assert(envelopePtr != NULL);
    Printf("SendSMTP(%s,%s,%s);\r\n",envelopePtr->eight22FilenameArr,envelopePtr->toAddressArr,
        envelopePtr->fromAddressArr);
    SendSMTP(envelopePtr);
    SetStatusText(""); // status.c
    PostMessage(hMainWnd,MISH_MAIL_STATUS,(WPARAM)envelopePtr,0);
    return 0;
}


