/* Copyright (c) Rick Dean 1997 */
/* <mishawaka@fdd.com> Rick Dean */

#include <windows.h>
#include <assert.h>
#include <stdio.h>
#include <commctrl.h>   
#include "schedule.h"
#include "outbound.h"
#include "mishawaka.h"
#include "debug.h"
#include "match.h"

//  This file is NOT re-entrant.  There is one thread for
//  the interface, and that is the only one which calls schedule.c

void SafeGlobalFree(void *voidPtr)
{
    if(voidPtr)
        GlobalFree(voidPtr);
};

static int ComplainMagicNumber(Envelope *envelopePtr)
{
    if(envelopePtr->magicNumber != ENVELOPE_MAGIC_NUMBER) { 
        Printf("magic number missing in queue flie\r\n");
        return 1;
    };
    return 0;
}

static HANDLE OpenQueueFile(void)
{
    HANDLE queueFileHdl;

    queueFileHdl = CreateFile("mishawaka queue.evl",GENERIC_READ|GENERIC_WRITE,FILE_SHARE_READ,NULL,OPEN_ALWAYS,FILE_FLAG_RANDOM_ACCESS,NULL);
    if(queueFileHdl == INVALID_HANDLE_VALUE) {
        Msg("Could not open existing queue file.\n");
        exit(1);
    };
    return queueFileHdl;
}

static void UpdateEnvelopeListing(Envelope *envelopePtr)
{
	LV_ITEM listViewItem;        // List view item structure
    char *statusStrPtr;
    LV_FINDINFO listViewFindInfo;
    extern int pauseNewSending;
    extern HWND hWndList;

    if(hWndList == NULL)
        return;
    listViewFindInfo.flags = LVFI_PARAM;
    listViewFindInfo.lParam = envelopePtr->queueFileOffset;
    listViewItem.iItem = ListView_FindItem(hWndList,-1,&listViewFindInfo);  // get index
    if(listViewItem.iItem == -1) {  // if item is not listed
        if(envelopePtr->status == EMPTY)
            return;
        listViewItem.mask = LVIF_IMAGE | LVIF_PARAM | LVIF_STATE;
        listViewItem.state = 0;      
        listViewItem.stateMask = 0xffffffff;  
        listViewItem.iSubItem = 0;  // neccessiary?
        listViewItem.lParam = envelopePtr->queueFileOffset;
        listViewItem.iImage = 0;
        listViewItem.iItem = 1;  // strangely -1 is a problem. Maybe is for initial position?
        listViewItem.iItem = ListView_InsertItem(hWndList,&listViewItem);
        if(listViewItem.iItem == -1)
            Printf("insert error (index=%d)\r\n",listViewItem.iItem);
    } else if(envelopePtr->status == EMPTY) {
        ListView_DeleteItem(hWndList,listViewItem.iItem);
        return;
    };
    ListView_SetItemText(hWndList,listViewItem.iItem,0,envelopePtr->eight22FilenameArr);
    ListView_SetItemText(hWndList,listViewItem.iItem,1,envelopePtr->fromAddressArr);
    ListView_SetItemText(hWndList,listViewItem.iItem,2,envelopePtr->toAddressArr);
    switch(envelopePtr->status) {
    case WAITING:
        if(pauseNewSending)
            statusStrPtr = "paused";
        else
            statusStrPtr = "waiting";
        break;
    case SENDING:
        statusStrPtr = "sending";
        break;
    case KEEPER:
        statusStrPtr = "keeper";
        break;
    default:
        statusStrPtr = "unknown";
        break;
    };
    ListView_SetItemText(hWndList,listViewItem.iItem,3,statusStrPtr);
}

void ResetListView(HWND hWndList)
{
    Envelope envelope;   
    int numBytesRead;
    HANDLE queueFileHdl;

    if(hWndList == NULL)
        return;
    ListView_DeleteAllItems(hWndList);
    queueFileHdl = OpenQueueFile();
    for(;;) {
        ReadFile(queueFileHdl,&envelope,sizeof(Envelope),&numBytesRead,NULL);
        if(numBytesRead == 0)
            break;
        ComplainMagicNumber(&envelope);
        UpdateEnvelopeListing(&envelope);
    };
    CloseHandle(queueFileHdl);
}

int IsAnyMail(Status status)
{
    Envelope envelope;   
    int numBytesRead,isAnyMail;
    HANDLE queueFileHdl;

    isAnyMail = FALSE;
    queueFileHdl = OpenQueueFile();
    for(;;) {
        ReadFile(queueFileHdl,&envelope,sizeof(Envelope),&numBytesRead,NULL);
        if(numBytesRead == 0)
            break;
        ComplainMagicNumber(&envelope);
        if(envelope.status == status) {
            isAnyMail = 1;
            break;
        };
    };
    CloseHandle(queueFileHdl);
    return isAnyMail;
};

//  This will not wrap until year 5600 something.
//  Are the extra calculations worth catching the next divisible-by-4-but-non-leap year in 2100 ?
static unsigned long MinutesSinceYear1600(void)
{
    SYSTEMTIME systemTime;
    volatile unsigned int daysSinceYear1600; // so the optimizer doesn't change the rounding
    static unsigned int daysForMonthArr[] = {0,31,59,90,120,151,181,212,243,273,304,334};
    unsigned int lastYear;

    GetSystemTime(&systemTime);  // in UMT so daylight savings is not a problem
    assert(systemTime.wYear > 100);
    lastYear = systemTime.wYear-1-1600; 
    daysSinceYear1600 = lastYear*365u + daysForMonthArr[systemTime.wMonth-1] + systemTime.wDay-1; 
    daysSinceYear1600 += lastYear/4 + 1;  // leap years before current year which are divisible by 4
    daysSinceYear1600 -= lastYear/100;  // but not divisible by 100
    daysSinceYear1600 += lastYear/400;  // unless are divisible by 400
    if(systemTime.wYear%4 == 0 && (systemTime.wYear%100 != 0 || systemTime.wYear%400 == 0) && systemTime.wMonth > 2)
        daysSinceYear1600++;       // leap day in current year
    return daysSinceYear1600*24u*60u + systemTime.wHour*60u + systemTime.wMinute;
}

static void OverwritePreviousEnvelope(HANDLE fileHdl,Envelope *envelopePtr)
{
    int numBytesWritten;

    SetFilePointer(fileHdl,-(signed)sizeof(Envelope),NULL,FILE_CURRENT);
    WriteFile(fileHdl,envelopePtr,sizeof(Envelope),&numBytesWritten,NULL);
    assert(numBytesWritten == sizeof(Envelope));
    FlushFileBuffers(fileHdl);
}

static void DeleteIfUnreferenced(HANDLE fileHdl,char *filenameArr)
{
    Envelope envelope;   
    int numBytesRead;

    SetFilePointer(fileHdl,0,NULL,FILE_BEGIN);
    for(;;) {
        ReadFile(fileHdl,&envelope,sizeof(Envelope),&numBytesRead,NULL);
        if(numBytesRead == 0) // if end of file
            break;
        ComplainMagicNumber(&envelope);
        if(envelope.status == EMPTY) 
            continue;
        if(strcmp(filenameArr,envelope.eight22FilenameArr) == 0)  // if referenced
            return;  
    };
    DeleteFile(filenameArr);
}

//  This routine will try to initiate a thread which will send a letter
//  This routine runs on the single GUI thread, so it can read/write to the queue file.
int DoScheduleTimer(void)
{
    Envelope *envelopePtr;
    int numBytesRead,threadID,currentMinute,minuteOfIdlestLetter = 0x7fffffff;
    HANDLE queueFileHdl;
    extern int minutesToWaitBeforeResending,pauseNewSending;
    extern HWND hMainWnd;

    KillTimer(hMainWnd,TIMER_INDEX_FOR_SCHEDULE);
    Printf("%d DoScheduleTimer()\r\n",MinutesSinceYear1600());
    if(pauseNewSending)
        return 0;
    queueFileHdl = OpenQueueFile();
    SetFilePointer(queueFileHdl,0,NULL,FILE_BEGIN);
    envelopePtr = GlobalAlloc(GMEM_FIXED,sizeof(Envelope));  
    if(envelopePtr == NULL)
        return 0;  // very low on memory???  
    currentMinute = MinutesSinceYear1600();
    for(;;) { // for every envelope in the queue file until we find one to send
        ReadFile(queueFileHdl,envelopePtr,sizeof(Envelope),&numBytesRead,NULL);
        if(numBytesRead == 0)  // if end of queue file 
            break;  
        ComplainMagicNumber(envelopePtr);
        if(envelopePtr->status != WAITING)
            continue;
        if(envelopePtr->minuteOfLastDeliveryAttempt+minutesToWaitBeforeResending > currentMinute) {
            if(minuteOfIdlestLetter > envelopePtr->minuteOfLastDeliveryAttempt)
                minuteOfIdlestLetter = envelopePtr->minuteOfLastDeliveryAttempt;
        } else {
            envelopePtr->status = SENDING;
            OverwritePreviousEnvelope(queueFileHdl,envelopePtr);
            UpdateEnvelopeListing(envelopePtr);
            if(CreateThread(NULL,0,(void*)OutboundThreadMain,envelopePtr,0,&threadID) == NULL) { // outbound.c
                Printf("Could not create outbound thread\r\n");
                break;
            };
            break;
        };
    };
    if(minuteOfIdlestLetter <= currentMinute) {
        Printf("minutesToWait=%d\r\n",(minutesToWaitBeforeResending-(currentMinute-minuteOfIdlestLetter)+1));
        SetTimer(hMainWnd,TIMER_INDEX_FOR_SCHEDULE,(minutesToWaitBeforeResending-(currentMinute-minuteOfIdlestLetter)+1)*60*1000,(void*)DoScheduleTimer);
    };
    CloseHandle(queueFileHdl);
    return 0; //  return "message processed" if WM_TIMER message
}

// According to RFC822, "unfolding" is removing the CRLF before continuation lines.
// "Continuation lines" are ones that begin with white space.
// The return value is the length of the line including the character null. 
// The trailing newline is replaced with a character null.
// numBytesInBuffer is returned if there are not enough characters (at least one after newline).
static int UnfoldLineAndGetNumBytes(char *bufferPtr,int numBytesInBuffer)
{
    char *crCharPtr;

    bufferPtr[numBytesInBuffer] = '\0';
    crCharPtr = bufferPtr;
    for(;;) {  // for every attempt at finding the end
        crCharPtr = strchr(crCharPtr,'\r');
        if(crCharPtr == NULL || crCharPtr-bufferPtr > numBytesInBuffer-2)
            return numBytesInBuffer;
        if(crCharPtr[1]=='\n')
            if(crCharPtr[2] ==' ' || crCharPtr[2]=='\t') 
                crCharPtr[0] = crCharPtr[1] = crCharPtr[2] = ' ';
            else 
                return crCharPtr-bufferPtr + 2;
        else
            return crCharPtr-bufferPtr + 1;  // actually an error
    };
}

static int WasReceivedTwice(char *filenamePtr)
{
    HANDLE fileHdl;
    extern char *myHostnamePtr;
    char bufferArr[1024],*tokenPtr;
    int numBytesInBuffer,numBytesInCurrentLine,answer,numBytesRead;

    fileHdl = CreateFile(filenamePtr,GENERIC_READ,FILE_SHARE_READ|FILE_SHARE_WRITE,NULL,OPEN_EXISTING,FILE_FLAG_SEQUENTIAL_SCAN,NULL);
    assert(fileHdl != INVALID_HANDLE_VALUE);
    answer = -1;  // allow for the received line we appended this time
    for(numBytesInBuffer=numBytesInCurrentLine=0;;) {  
        if(numBytesInCurrentLine > 0)
            memmove(bufferArr,bufferArr+numBytesInCurrentLine,numBytesInBuffer-numBytesInCurrentLine);
        numBytesInBuffer -= numBytesInCurrentLine;
        numBytesInCurrentLine = UnfoldLineAndGetNumBytes(bufferArr,numBytesInBuffer);
        if(numBytesInCurrentLine == numBytesInBuffer) { 
            ReadFile(fileHdl,bufferArr+numBytesInBuffer,sizeof(bufferArr)-numBytesInBuffer-1,&numBytesRead,NULL);
            numBytesInBuffer += numBytesRead;
            numBytesInCurrentLine = UnfoldLineAndGetNumBytes(bufferArr,numBytesInBuffer);
        };
        if(numBytesInCurrentLine <= 2)
            break;  // end of search
        bufferArr[numBytesInCurrentLine-1] = '\0';
        if(strncmp(bufferArr,"Received:",9) != 0) 
            continue;
        for(tokenPtr=strtok(bufferArr/*+9*/," \t");tokenPtr;tokenPtr = strtok(NULL," \t")) {
            if(strcmp(tokenPtr,"by") != 0) // if token is not "by"
                continue;
            if((tokenPtr = strtok(NULL," \t")) == NULL) // if no more tokens
                break;
            if(strcmp(tokenPtr,myHostnamePtr) == 0) {
                answer++;
                if(answer > 0)
                    break;
            };
        };
    };   
    CloseHandle(fileHdl);
    assert(answer != -1);
    return answer; 
}

//  This routine is recursive whenever an applicable alias rule has multiple recipients.
static void RouteEnvelope(HANDLE queueFileHdl,Envelope *envelopePtr)
{
    extern char *keepEAddressList;

    envelopePtr->minuteOfReceipt = MinutesSinceYear1600();
    envelopePtr->minuteOfLastDeliveryAttempt = 0;
    if(WasReceivedTwice(envelopePtr->eight22FilenameArr)) {
        envelopePtr->status = KEEPER;
        return;
    };
    if(IsMatchList(keepEAddressList,envelopePtr->toAddressArr)) 
        envelopePtr->status = KEEPER;
    else            
        envelopePtr->status = WAITING;
}

static void ChangeEnvelopeForBounce(HANDLE queueFileHdl,Envelope *envelopePtr,char *reasonPtr)
{
    if(reasonPtr == NULL)
        reasonPtr = "";
    if(strcmp(envelopePtr->fromAddressArr,"<>") == 0) {//if no return address (already from a bounce)
        envelopePtr->status = EMPTY;
        return;
    };
    strcpy(envelopePtr->toAddressArr,envelopePtr->fromAddressArr);
    strcpy(envelopePtr->fromAddressArr,"<>"); // remove return address
    envelopePtr->status = WAITING;
    envelopePtr->minuteOfLastDeliveryAttempt = 0;
    // need to include error message in body
    RouteEnvelope(queueFileHdl,envelopePtr);
}

static void DoViewContents(char *filenameArr)
{ 
    STARTUPINFO startupInfo;
    PROCESS_INFORMATION processInfomation;
    char commandArr[1024];

    sprintf(commandArr,"notepad %s",filenameArr);
    startupInfo.cb = sizeof(STARTUPINFO);
    startupInfo.lpReserved = NULL;
    startupInfo.lpDesktop = NULL;
    startupInfo.lpTitle = NULL;
    startupInfo.dwFlags = 0;
    startupInfo.cbReserved2 = 0;
    startupInfo.lpReserved2 = NULL;
    CreateProcess(NULL,commandArr,NULL,NULL,FALSE,CREATE_DEFAULT_ERROR_MODE|NORMAL_PRIORITY_CLASS,
        NULL,NULL,&startupInfo,&processInfomation);
}

static BOOL CALLBACK EditDialogProc(HWND hWndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    int command;
    static Envelope *envelopePtr;
    
    switch (uMsg) {
    case WM_INITDIALOG:
        envelopePtr = (Envelope*)lParam;
        SetDlgItemText(hWndDlg,IDE_FROM_EADDRESS,envelopePtr->fromAddressArr);
        SetDlgItemText(hWndDlg,IDE_TO_EADDRESS,envelopePtr->toAddressArr);
        return 1; // return "did not set focus"
    case WM_COMMAND:
        command = LOWORD(wParam);
        if(command == IDOK) {
            GetDlgItemText(hWndDlg,IDE_TO_EADDRESS,envelopePtr->toAddressArr,sizeof(envelopePtr->toAddressArr));
            envelopePtr->toAddressArr[sizeof(envelopePtr->toAddressArr)-1] = '\0';
            GetDlgItemText(hWndDlg,IDE_FROM_EADDRESS,envelopePtr->fromAddressArr,sizeof(envelopePtr->fromAddressArr));
            envelopePtr->fromAddressArr[sizeof(envelopePtr->fromAddressArr)-1] = '\0';
        };
        if(command == IDCANCEL || command == IDOK) { 
            EndDialog(hWndDlg,0);
            return TRUE; 
        };
    }
    return FALSE;  // return "message not handled"
}


void DoCommandOnEnvleope(char *filenameArr,int queueFileOffset,int command)
{
    HANDLE queueFileHdl;
    Envelope envelope;
    int numBytesRead;
    extern HWND hMainWnd;

    assert(queueFileOffset >= 0);
    queueFileHdl = OpenQueueFile();
    SetFilePointer(queueFileHdl,queueFileOffset,NULL,FILE_BEGIN);
    ReadFile(queueFileHdl,&envelope,sizeof(Envelope),&numBytesRead,NULL);
    if(numBytesRead == 0)
        return;
    ComplainMagicNumber(&envelope);
    if(envelope.status == SENDING) {
        CloseHandle(queueFileHdl);
        return;
    };
    switch(command) {
    case IDM_VIEW_CONTENTS:
        DoViewContents(filenameArr);
        break;
    case IDM_DELETE_ENVELOPE:
        envelope.status = EMPTY;
        break;
    case IDM_KEEPER_ENVELOPE:
        envelope.status = KEEPER;
        break;
    case IDM_EDIT_ENVELOPE:
        DialogBoxParam(hInst,MAKEINTRESOURCE(IDD_EDITE),hMainWnd,(DLGPROC)EditDialogProc,(LPARAM)&envelope);
        break;
    case IDM_NEXT_ENVELOPE:
        envelope.status = WAITING;
        envelope.minuteOfLastDeliveryAttempt = 0;
        break;
    case IDM_USER_NOT_FOUND:
        ChangeEnvelopeForBounce(queueFileHdl,&envelope,"550 User not found");
        break;
    case IDM_UNDELIVERABLE:
        ChangeEnvelopeForBounce(queueFileHdl,&envelope,"500 Undeliverable");
        break;
    case IDM_FUCK_YOURSELF:
        ChangeEnvelopeForBounce(queueFileHdl,&envelope,"500 Go fuck yourself spammer!!!");
        break;
    };
    OverwritePreviousEnvelope(queueFileHdl,&envelope);
    DeleteIfUnreferenced(queueFileHdl,envelope.eight22FilenameArr);
    CloseHandle(queueFileHdl);
    UpdateEnvelopeListing(&envelope);
}

// This event is generated when outbound.c finishes sending a letter,
// It generates an event (instead of calling this directly) so that this
// function runs on the single interface thread to avoid re-entrant problems.
void DoMailStatus(HWND hWnd,Envelope *envelopePtr) 
{
    int numBytesWritten,numBytesRead;
    HANDLE queueFileHdl;
    Envelope actualEnvelope;
    extern int mintuesToWaitBeforeBouncing,returnBouncedMailToSender;

    queueFileHdl = OpenQueueFile();
    SetFilePointer(queueFileHdl,envelopePtr->queueFileOffset,NULL,FILE_BEGIN);
    ReadFile(queueFileHdl,&actualEnvelope,sizeof(Envelope),&numBytesRead,NULL);
    if(strcmp(envelopePtr->eight22FilenameArr,actualEnvelope.eight22FilenameArr)){//if not same envelope
        GlobalFree(envelopePtr);
        return;
    };
    switch(envelopePtr->status) {
        case SUCCESS_FILE_SENT:
            envelopePtr->status = EMPTY; // delete envelope
            Printf("mail success\r\n");
            break;
        case NETWORK_PROBLEM:
            envelopePtr->status = WAITING;
            envelopePtr->minuteOfLastDeliveryAttempt = MinutesSinceYear1600();
            if(envelopePtr->minuteOfLastDeliveryAttempt - envelopePtr->minuteOfReceipt > mintuesToWaitBeforeBouncing) {
                if(returnBouncedMailToSender)
                    ChangeEnvelopeForBounce(queueFileHdl,envelopePtr,"Undeliverable, destination not responding");
                else
                    envelopePtr->status = EMPTY; // delete envelope
            };
            Printf("##network problem\r\n");
            break;
        case MAIL_LOOP:
            Printf("##mail loop\r\n");
            envelopePtr->status = KEEPER;
            break;
        case SMTP_ERROR:
            ChangeEnvelopeForBounce(queueFileHdl,envelopePtr,envelopePtr->smtpErrorPtr);
            SafeGlobalFree(envelopePtr->smtpErrorPtr);
            Printf("##SMTP error\r\n");
            break;
        case DESTINATION_DOES_NOT_EXIST:
            ChangeEnvelopeForBounce(queueFileHdl,envelopePtr,"Destination does not exist");
            Printf("##destination DNE\r\n");
            break;
        default: // INTERNAL_ERROR:   
            envelopePtr->status = ERRORED;
            Printf("##internal error\r\n");
            break;
    };
    SetFilePointer(queueFileHdl,envelopePtr->queueFileOffset,NULL,FILE_BEGIN);
    WriteFile(queueFileHdl,envelopePtr,sizeof(Envelope),&numBytesWritten,NULL);
    FlushFileBuffers(queueFileHdl);
    DeleteIfUnreferenced(queueFileHdl,envelopePtr->eight22FilenameArr);
    CloseHandle(queueFileHdl);
    UpdateEnvelopeListing(envelopePtr);
    GlobalFree(envelopePtr);
    DoScheduleTimer();
    return;
}

// Move the file cursor.
// We go from the current position to the end, and start from the begging again.
// If we reach our starting point (file full), then we move to the end and return.
void MoveFileToEmptyEnvelope(HANDLE fileHdl)
{
    int startingPosition;
    Envelope envelope;   
    int numBytesRead;

    startingPosition = SetFilePointer(fileHdl,0,NULL,FILE_CURRENT);
    assert(startingPosition%sizeof(Envelope) == 0);
    for(;;) {
        ReadFile(fileHdl,&envelope,sizeof(Envelope),&numBytesRead,NULL);
        if(numBytesRead == 0)
            break;
        ComplainMagicNumber(&envelope);
        if(envelope.status == EMPTY) {
            SetFilePointer(fileHdl,-(signed)sizeof(Envelope),NULL,FILE_CURRENT);
            return;
        };
    };
    SetFilePointer(fileHdl,0,NULL,FILE_BEGIN);
    for(;startingPosition > 0;startingPosition -= sizeof(Envelope)) {
        ReadFile(fileHdl,&envelope,sizeof(Envelope),&numBytesRead,NULL);
        assert(numBytesRead == sizeof(Envelope));
        ComplainMagicNumber(&envelope);
        if(envelope.status == EMPTY) {
            SetFilePointer(fileHdl,-(signed)sizeof(Envelope),NULL,FILE_CURRENT);
            return;
        };
    };
    SetFilePointer(fileHdl,0,NULL,FILE_END);
}

//  This routine does no error processing.
static unsigned GetFileSizeByName(char *filenamePtr)
{
    unsigned fileSize;
    HANDLE fileHdl;

    fileHdl = CreateFile(filenamePtr,GENERIC_READ,FILE_SHARE_READ|FILE_SHARE_WRITE,NULL,OPEN_EXISTING,0,NULL);
    if(fileHdl == INVALID_HANDLE_VALUE)
        return 1;
    fileSize = GetFileSize(fileHdl,NULL);
    CloseHandle(fileHdl);
    return fileSize;
}

// This event is generated when inbound.c finishes receiving a letter,
// It generates an event (instead of calling this directly) so that this
// function runs on the single interface thread to avoid re-entrant problems.
// This routine appends the new (multiple) envelope file to the queue file.
// This routine finishes initilizing the envelope
void DoRouteInboundMail(HWND hWnd,char *filenamePtr)
{
    Envelope envelope;
    HANDLE newEnvelopeFileHdl;
    int numBytesRead,numBytesWritten,sizeOf822File;
    HANDLE queueFileHdl;

    queueFileHdl = OpenQueueFile();
    assert(filenamePtr);
    newEnvelopeFileHdl = CreateFile(filenamePtr,GENERIC_READ,0,NULL,OPEN_EXISTING,FILE_FLAG_SEQUENTIAL_SCAN,NULL);
    if(newEnvelopeFileHdl == INVALID_HANDLE_VALUE) {
        Printf("error opening %s\r\n",filenamePtr);
    } else {
        sizeOf822File = 0xfffffffe;
        for(;;) {  // for every envelope in file
            ReadFile(newEnvelopeFileHdl,&envelope,sizeof(Envelope),&numBytesRead,NULL);
            if(numBytesRead != sizeof(envelope)) {  // if end of file
                FlushFileBuffers(queueFileHdl);
                CloseHandle(newEnvelopeFileHdl);
                DeleteFile(filenamePtr);
                break;
            };
            ComplainMagicNumber(&envelope);
            MoveFileToEmptyEnvelope(queueFileHdl);
            envelope.queueFileOffset = SetFilePointer(queueFileHdl,0,NULL,FILE_CURRENT);
            RouteEnvelope(queueFileHdl,&envelope);
            if(sizeOf822File == 0xfffffffe)
                sizeOf822File = GetFileSizeByName(envelope.eight22FilenameArr);
            envelope.sizeOf822File = sizeOf822File;
            WriteFile(queueFileHdl,&envelope,sizeof(envelope),&numBytesWritten,NULL);
            if(numBytesRead != numBytesWritten) {
                Printf("disk full during write of %s to qeueu1.evl\r\n",filenamePtr);
                CloseHandle(newEnvelopeFileHdl);
                break;
            };
            UpdateEnvelopeListing(&envelope);
        };
    };
    GlobalFree(filenamePtr);
    CloseHandle(queueFileHdl);
    DoScheduleTimer();
}

//  Complain about magic numbers.
//  Make all SENDING envelopes WAITING.
void ResetQueueFile(HANDLE queueFileHdl)
{
    Envelope envelope;   
    int numBytesRead;

    SetFilePointer(queueFileHdl,0,NULL,FILE_BEGIN);
    for(;;) {
        ReadFile(queueFileHdl,&envelope,sizeof(Envelope),&numBytesRead,NULL);
        if(numBytesRead == 0)
            break;
        ComplainMagicNumber(&envelope);
        if(envelope.status == SENDING) {
            envelope.status = WAITING;
            OverwritePreviousEnvelope(queueFileHdl,&envelope);
        };
    };
}

void InitSchedule(HWND hWnd)
{
    HANDLE queueFileHdl;

    queueFileHdl = OpenQueueFile();
    ResetQueueFile(queueFileHdl);
    //FlushFileBuffers(queueFileHdl);  // can't be any worse than before we started
    CloseHandle(queueFileHdl);
    DoScheduleTimer();
}

void QuitSchedule(HWND hWnd)
{
}

