
# Copyright (c) 2003 Frederick Dean

use strict;

package card;

use Gtk;
use Gtk::Gdk::Pixbuf;

# This module is all about drawing cards, but not where.

#  Cards have two letter names.
#  c=club     a=ace 
#  d=diamons  2=2
#  h=hearts   3=3 
#  s=spades   4=4  
#             t=10
#             j=jack
#             q=queen

my %images;     # The key is the filename.
my %mask;
my %card_cmds;  # This is a list of drawing commands.
                # The key is the two character card name.
                # Each drawing command is a list of [ $image, $from_x, $from_y, $to_x, $to_y, $width, $height ]

our @suits = qw/S H C D/;
our %suithash = ( "S" => 0, "H" => 1, "C" => 2, "D" => 3 );
our @ranks = qw/A K Q J T 9 8 7 6 5 4 3 2/;
our(@cards,%cardsort);
foreach my $suit (@suits) { 
   foreach my $rank (@ranks) { 
      $cardsort{"$suit$rank"} = $#cards + 1;
      push(@cards,"$suit$rank");
   }
}
$cardsort{'BACK'} = 101;
my $specialname;  # name of altername first image for special card.

sub load_image
{
   my($imagename) = @_;
   return if defined $images{$imagename}; # if already loaded
   print("loading $imagename\n") if $::debug;
   die "Imaage $imagename does not exist." if ! -e $imagename;
   die "Imaage $imagename is not readable (file permissions)." if ! -r $imagename;
   $images{$imagename} = Gtk::Gdk::Pixbuf->new_from_file( $imagename );
   $mask{$imagename} = $images{$imagename}->render_pixmap_and_mask(127);
   defined $images{$imagename} or die "could not load image $imagename: $!\n";
}

sub load_deck
{
   my($deckname) = @_;

   Gtk::Gdk::Pixbuf->init();
   print("loading deck $deckname\n") if $::debug;
   -e $deckname or die "filename $deckname does not exist\n";
   -r $deckname or die "filename $deckname is not readable\n";
   open(DECK,"<$deckname") or die "cannot open $deckname: $!\n";

   my $card;
   my $line_num = 0;
   while(my $line = <DECK>) {  # for every line of deck description
      $line_num++;
      chomp($line);  # remove trailing newline
      $line =~ s/\s*#.*//;  # delete after hash (and space immediately before)
      next if $line =~ /^\s*$/;  # skip line if blank 
      #print("---> $line\n");
      my @tokens = split(/\s+/,$line);
      if($tokens[0] =~ /^card$/i) {  # if it is a card line
          $card = uc($tokens[1]);
          defined $cardsort{$card} or die "bad card name ($card): $deckname line $line_num\n";
          defined $card_cmds{$card} and die "duplicate card ($card): $deckname line $line_num\n"; 
      } elsif($tokens[0] =~ /^image$/i) {  # if it is an image line
          defined $card or die "image line before card: $deckname line $line_num\n";
          my($imagename,$src_x,$src_y,$dest_x,$dest_y,$width,$height) = @tokens[1..7];
          load_image($imagename);
          $src_x >= 0 or die "src_x negative: $deckname line $line_num\n";
          $src_y >= 0 or die "src_x negative: $deckname line $line_num\n";
          $dest_x >= 0 or die "dest_y negative: $deckname line $line_num\n";
          $dest_y >= 0 or die "dest_y negative: $deckname line $line_num\n";
          $width = $images{$imagename}->get_width() if $width eq "*";  # auto width
          $height = $images{$imagename}->get_height() if $height eq "*";  # auto height
          $src_x < $images{$imagename}->get_width() or die "src_x bigger than image width: $deckname line $line_num\n";
          $src_y < $images{$imagename}->get_height() or die "src_y bigger than image height: $deckname line $line_num\n";
          $src_x + $width <= $images{$imagename}->get_width() or die "src_x+width larger than image width: $deckname line $line_num\n";
          $src_y + $height <= $images{$imagename}->get_height() or die "src_y+height larger than image height: $deckname line $line_num\n";
          push(@{$card_cmds{$card}},[ $imagename, $src_x,$src_y,$dest_x,$dest_y,$width,$height ]);
      } elsif($tokens[0] =~ /^special$/i) {  # if it is an image line
          $specialname = $tokens[1];
          load_image($specialname);
      };
   }
   close(DECK);
   foreach (keys %cardsort) {
      die "missing card $_ in $deckname\n" if ! defined $card_cmds{$_};      
   }
}

#  $widget is a Gtk::Window or Gtk::Widget not Gdk::Window
#  $card is a two letter card code string.
#  If $show_height is given then we clip the bottom of the card.
sub draw
{
   my($widget,$card,$x,$y,$show_height,$special) = @_;

   my $gc = Gtk::Gdk::GC->new($widget->window);
   defined $card_cmds{$card} or die "undefined card $card";
   my @cmds = @{$card_cmds{$card}};
   foreach my $cmdref (@{$card_cmds{$card}}) {
      my($imagename,$src_x,$src_y,$dest_x,$dest_y,$width,$height) = @$cmdref;
      $imagename = $specialname if $special && $specialname;
      next if $show_height && $dest_y >= $show_height;
      $height = $show_height - $dest_y if $show_height && $height > $show_height - $dest_y;
      if(defined $mask{$imagename}) {  # if image has a mask
         $gc->set_clip_mask($mask{$imagename});
         $gc->set_clip_origin($dest_x + $x - $src_x,$dest_y + $y - $src_y);
      } else {  # else image does not have an alpha
         $gc->set_clip_rectangle({ x=>0, y=>0, width=>2000, height=>2000 }); # basically everything
      }
      $images{$imagename}->render_to_drawable($widget->window,$gc,$src_x,$src_y,$dest_x + $x,$dest_y + $y,$width,$height);
      $special = undef;
   }
}

sub arrow  # ugly
{
   my($widget,$dir,$x,$y) = @_;
   my $gc2 = Gtk::Gdk::GC->new($widget->window);
   $widget->window->draw_polygon($gc2,1, $x+10,$y+0,  $x+20,$y+0,  $x+20,$y+10, $x+30,$y+10, $x+15,$y+20, $x+0,$y+10, $x+10,$y+10) if $dir eq 'N';
   $widget->window->draw_polygon($gc2,1, $x+10,$y+20, $x+20,$y+20, $x+20,$y+10, $x+30,$y+10, $x+15,$y+0,  $x+0,$y+10, $x+10,$y+10) if $dir eq 'S';
   #$widget->window->draw_polygon($gc2,1, $x+20,$y+10, $x+20,$y+20, $x+10,$y+20, $x+10,$y+30, $x+0,$y+15,  $x+10,$y+0, $x+10,$y+10) if $dir eq 'E';
}

1;
