#!/usr/bin/perl -w
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Get it at http://www.remote-exploit.org
#
# Copyright (c) Frederick Dean 2003
#
use strict;

BEGIN {  # Search for my files.
   my $my_file_path = "/path/to/my/files/";
   if(-e "$my_file_path/bid.pm") {  # check path 
      unshift(@INC,$my_file_path);
   } elsif($0 =~ /(.*)\//s && -e "$1/bid.pm") {  # check path of excutable
      unshift(@INC,$1);
   } elsif (-e "./bid.pm") {  # check current directory
      unshift(@INC,".");
   } else {
      die "I cannot find my files like bid.pm.\n".
          "Please edit the \$my_file_path line in $0\n";
   };
   if(! -r "$INC[0]/bid.pm") {  # if I cannot read bid.pm
     die qq{### Cannot read "$INC[0]/bid.pm".\n### You have a file permission problem.\n### Please fix with "chmod a+r $INC[0]/*.pm".\n};
   }
}
chdir($INC[0]) if $INC[0] ne ".";  # change current directory so deck images load okay

require 5.006;  # for the "our" stuff

# Silence repeated undef warnings
$SIG{__WARN__} = sub { warn @_ unless $_[0] =~ /Use of uninitialized value in subroutine entry at $0/ };

use Gtk;
use Gtk::Gdk::ImlibImage;
use card; 
use deal;
use bid;
use score;
use pbn;
use MD5;
use Carp;
use history;

our $debug = 1;  # flag
print qq{Using modules in "$INC[0]"\n} if $debug;

$SIG{INT} = \&conniption;

Gtk->set_locale;
Gtk->init;

# These globals are useful for displaying stuff
our $main_window;
my(%center_widgets,%center_keystrokes);  # deal, bid, play, score
my $curr_center;  # "deal", "bid", "play", or "score"
my($da_west,$da_east,$da_north,$da_south,$da_play);  # drawing areas
my %exposures = ( W => \&expose_drawingarea_west, E => \&expose_drawingarea_east, 
                  S => \&expose_drawingarea_south, N => \&expose_drawingarea_north );
our $status_label;  #  Gtk::Label   (left statusbar)
our $dealer_label;
our $contract_label;
my $nswins_label;
my $ewwins_label;

# Some useful constants
our %full_dir =     (N => "North", E => "East", S => "South", W => "West");
our %to_left =      (N => "E",     E => "S",    S => "W",     W => "N");
our %to_right =     (N => "W",     E => "N",    S => "E",     W => "S");
our %to_partner =   (N => "S",     E => "W",    S => "N",     W => "E");

# Some game state
our %where;  # The 52 keys are the cards.  SA SK SQ SJ ST S9 S8 ...      (also @card::cards)
             # The value ("when") is a string whos first letter indicates a player N, S, E, or W
             # The remainer of the string is the turn number the card was played, or "-" if unknown/not-yet.
             # Use played() for the reverse mapping of this hash.
our $dealer = "S";  # will advance one before first deal
our $contract = "--";  # 2S, 3C, 4NT, etc.
our $double = 1;  # 1=none, 2=doubled, 4=redoubled
our $declarer = "--";  # N, E, etc.  (person of bid-winnnig team who first mentioned winning suit)
our $show_hands = "S";  # search for your letter
our($ns_tricks,$ew_tricks) = (0,0);  # won
our $whos_turn = "";  # N, S, E, W     The current person waiting to play.
our $trick = 0;  #   Number of.  Starts with 1 
our @lead;   #   Who lead each trick $lead[$trick].  Also show current trick winner $lead[$trick+1]
our $lead_suit = "";  #  S, N, E, W
my $chosen_card = -1;  # highlighted on screen.  Use the arrow keys.  Press the space bar.
my @card_click_this_trick;  # for undo. $card_click_this_trick[$trick] where $trick starts with 1 (afterglow does not count)

sub sorted_hand
{ 
   my($who,$andplayed) = @_;   # $andplayed includes all the cards they were dealt even if already played
   my @hand = grep { substr($where{$_},0,1) eq $who && ($andplayed || $where{$_} =~ /-$/)} @card::cards;
   @hand = sort { $card::cardsort{$a} <=> $card::cardsort{$b} } @hand;
   return @hand;
}

# Kind of like die, but with more information
sub conniption 
{
   print "*-" x 40 ."\n";
   print "trick=$trick chosen_card=$chosen_card lead_suit=$lead_suit whos_turn=$whos_turn dealer=$dealer\n".
         "contract=$contract double=$double declarer=$declarer ns_tricks=$ns_tricks ew_tricks=$ew_tricks\n";
   foreach my $t (0..14) {
      $lead[$t] = "undef" if ! defined $lead[$t];
      $card_click_this_trick[$t] = "undef" if ! defined $card_click_this_trick[$t];
      print " $t) played: ".
            " W=". (join(".",grep { $where{$_} eq "W$t" } @card::cards) ||"  ") .  # ugly
            " N=". (join(".",grep { $where{$_} eq "N$t" } @card::cards) ||"  ") .
            " E=". (join(".",grep { $where{$_} eq "E$t" } @card::cards) ||"  ") .
            " S=". (join(".",grep { $where{$_} eq "S$t" } @card::cards) ||"  ") .  
            " cctt=$card_click_this_trick[$t] lead=$lead[$t]\n"; 
   }
   print "west  where: ". join(" ",map { "$_->$where{$_}" } sorted_hand('W',1)) ."\n";
   print "north where: ". join(" ",map { "$_->$where{$_}" } sorted_hand('N',1)) ."\n";
   print "east  where: ". join(" ",map { "$_->$where{$_}" } sorted_hand('E',1)) ."\n";
   print "south where: ". join(" ",map { "$_->$where{$_}" } sorted_hand('S',1)) ."\n";
   print "west hand:  ". join(".",sorted_hand('W',0)) ."\n";
   print "north hand: ". join(".",sorted_hand('N',0)) ."\n";
   print "east hand:  ". join(".",sorted_hand('E',0)) ."\n";
   print "south hand: ". join(".",sorted_hand('S',0)) ."\n";
   Carp::confess(@_);
}

#  Tell what card was played on that turn.
#  This is the reverse mapping of %where.
#  Sure we could keep track of another hash, but performance is fine and this is easier. 
sub played
{
  my($when) = @_;  #   N2, E3, W10, etc.
  my @card = grep { $where{$_} eq  $when } @card::cards;
  conniption if scalar(@card) > 1;  # if more than one card matches
  return $card[0];
}

# Everyone calls this.
sub play_card
{
   my($card) = @_;
   conniption("Cannot play false card") unless $card;  # can't play undefined card
   conniption("Cannot play non-existent card") unless defined $where{$card}; 
   conniption("Can only play card in my hand (not at $where{$card})") unless $where{$card} eq $whos_turn . "-"; 
   conniption("Cannot play twice this trick") if played($whos_turn . $trick); 
   $chosen_card = -1;  # For keyboard choice of the card
   $where{$card} = $whos_turn . $trick; # play the card
   history::report_turn($card,$trick,$whos_turn);
   &{$exposures{$whos_turn}};  # redraw the hand with one less card
   $lead_suit = $lead_suit || substr($card,0,1);
   # compute the winning card this trick so far
   my $best = 0;
   foreach my $card_this_trick (grep { substr($where{$_},1) eq $trick } @card::cards) {  # for each card of this trick
      my $score = 200*(substr($card_this_trick,0,1) eq $lead_suit) + 
                  400*(substr($card_this_trick,0,1) eq substr($contract,1)) - $card::cardsort{$card_this_trick};
      if($score > $best) {  # if better than the best so far
         $best = $score;
         $lead[$trick+1] = substr($where{$card_this_trick},0,1);  # determine current trick winner so far
      }
   }
   conniption("What no winning card this trick?") if ! defined $lead[$trick+1];
my $x = scalar(sorted_hand($whos_turn));
   conniption("Wrong number of cards for $whos_turn") if  scalar(sorted_hand($whos_turn)) != 13 - $trick; 
   $whos_turn = $to_left{$whos_turn}; # advance the turn
   if($whos_turn eq $lead[$trick]) {  # if everyone has played
      $whos_turn = "";
      if($lead[$trick+1] eq "N" || $lead[$trick+1] eq "S") {
         $ns_tricks++;
         $nswins_label->set("N/S Wins: $ns_tricks");
      } else {
         $ew_tricks++;
         $ewwins_label->set("E/W Wins: $ew_tricks");
      }
   };
   expose_drawingarea_play();  # redraw the center
   expose_drawingarea_east() if($trick == 1 && $whos_turn eq "S" && $declarer eq "E"); # if first card is played when South declarer
}

#  Play one card
sub play_turn_ai
{
   require play_ai;
   return play_card(play_ai::play_turn_ai());
}

# Will play for AI if it can.
sub play_ai_while_we_can
{
   while(1) {  # while we can play card without the human
      if($lead_suit) {  # if the trick has already been lead
         my @choices = grep { substr($_,0,1) eq $lead_suit } sorted_hand($whos_turn);  # determine hands legal choices
         if(scalar(@choices) == 1) {  # if only one legally playable card
            play_card($choices[0]);
            next;
         }
      }
      if($trick == 13 && $whos_turn) {  # thirteenth trick has only one choice
         my @hand = sorted_hand($whos_turn);  # determine hands legal choices
         play_card($hand[0]);
         next;
      }
      last if $whos_turn eq "S";  # if human's turn,  stop autoplaying
      last if $whos_turn eq "N" && $declarer eq "S"; # if human's dummy's turn,  stop autoplaying
      last if $whos_turn eq "";  # if trick is complete 
      play_turn_ai();
   }
   if($whos_turn) {
      $status_label->set($full_dir{$whos_turn} ."'s turn.  Pick a card.");
   } else { 
      $status_label->set("Spacebar or mouse-click for next trick.") if $trick < 14;
   }
}

#  Trick 1 means no cards played.
sub undo_to_trick
{
   my($to_trick) = @_;
   $to_trick = 0 if $contract !~ /^\d/;
   history::erase_trick_and_after($to_trick);
   # restore cards to the hands
   foreach my $dir (qw/N S E W/) {  # for every direction
      foreach my $t ($to_trick..$trick) {  # for every turn which might need undoing
         my $card = played("$dir$t");
         if($card) {  # if a card was played
            $where{$card} = $dir ."-"; # remove from play list
         }
      }
      if($to_trick && scalar(sorted_hand($dir)) != 14 - $to_trick) { #check right number of cards
         conniption "Wrong number of unplayed cards (". scalar(sorted_hand($dir)) .") in $full_dir{$dir} hand"; 
      };
   }
   foreach my $t ($to_trick..13) {  # for every turn which might need undoing
      next if $t < 1;
      if($trick > $t || ($trick == $t && $whos_turn eq "")) {  # if the trick was completed
          $ns_tricks-- if($lead[$t+1] eq 'N' || $lead[$t+1] eq 'S');  # if North or South won, uncount it
          $ew_tricks-- if($lead[$t+1] eq 'E' || $lead[$t+1] eq 'W');  # if North or South won, uncount it
      }
   }
   $nswins_label->set("N/S Wins: $ns_tricks");
   $ewwins_label->set("E/W Wins: $ew_tricks");
   $trick = $to_trick;
   $card_click_this_trick[$trick] = 0;
   $whos_turn = $lead[$trick];
   $lead_suit = "";
   if($trick < 1) {
      show_center('bid');
      $::status_label->set("Click Play.");
      $declarer = "";   # so the dummy does not show
      if($bid::decl eq 'N') {  # if the hands were rotated after bidding
         foreach (keys %::where) {  # rotate hands 180 degrees
            $where{$_} = $::to_partner{substr($::where{$_},0,1)} . substr($::where{$_},1);
         }
      };
      expose_drawingareas();
   } else {
      expose_drawingareas();
      play_ai_while_we_can();
   };
}

sub do_undo 
{ 
   if($curr_center eq "score") {
      show_center('play');
      undo_to_trick(13); 
   } elsif($curr_center eq "play") {
      if(! $card_click_this_trick[$trick]) {  # if the current trick is incomplete (or trivial)
         undo_to_trick($trick-1); 
      } else {  # else the current trick is complete
         undo_to_trick($trick); 
      }
   } elsif($curr_center eq "bid") {
      bid::bid();
   };
}

sub do_auto 
{ 
   if($curr_center eq 'play' && $trick > 0 && $trick < 14) {
      if($whos_turn eq "S" || ($whos_turn eq 'N' && $declarer eq "S")) {  # if humans turn
         $card_click_this_trick[$trick] = 1;
         play_turn_ai();
         play_ai_while_we_can();
      } elsif ($whos_turn eq "") {  # else if end of trick
         do_play_spacebar();
      }
   } elsif($curr_center eq 'bid') {
      bid::auto_clicked();
   }
};

sub do_hint 
{
   if($curr_center eq 'play') {
      end_trick_afterglow();
      if($trick < 14 && ($whos_turn eq "S" || ($whos_turn eq 'N' && $declarer eq "S"))) { 
         require play_ai;
         my $hint_card = play_ai::play_turn_ai();
         my @hand = sorted_hand($whos_turn);
         foreach (0..$#hand) { $chosen_card = $_ if $hand[$_] eq $hint_card };
         &{$exposures{$whos_turn}};  # redraw hand
      }
   } elsif($curr_center eq 'score') {
      $status_label->("Play better!");
   } elsif($curr_center eq 'bid') {
      bid::hint_clicked();
   }
}

sub start_play
{
   $trick = 1;
   return score::score() if($contract !~ /^\d/);  # if contract doesn't begin with a number
   ::show_center('play'); 
   history::erase_trick_and_after(1);
   ($ns_tricks,$ew_tricks) = (0,0);
   $nswins_label->set("N/S Wins: $ns_tricks");
   $ewwins_label->set("E/W Wins: $ew_tricks");
   if($declarer eq "N") {  # if north is declarer
      foreach (keys %::where) {  # rotate hands 180 degrees
         $where{$_} = $::to_partner{substr($::where{$_},0,1)} . substr($::where{$_},1);
      }
      $declarer = "S";
      ::expose_drawingareas();   # redraw the screen
   }
   $whos_turn = $to_left{$declarer};
   @lead = ("",$whos_turn);
   $lead_suit = "";
   @card_click_this_trick = (0,0);
   play_ai_while_we_can();
}

sub play_key_left 
{
   return unless $whos_turn;
   $chosen_card--;
   my @hand = sorted_hand($whos_turn);
   $chosen_card = $#hand if $chosen_card < 0;
   if($lead_suit && scalar(grep { substr($_,0,1) eq $lead_suit } @hand) &&  # if we have to follow suit
      substr($hand[$chosen_card],0,1) ne $lead_suit) {  # and we were selecting the wrong suit
      $chosen_card = $#hand;  # start with last card
      while(substr($hand[$chosen_card],0,1) ne $lead_suit) {
        $chosen_card--;
      }
   }
   &{$exposures{$whos_turn}};
}
sub play_key_right 
{
   return unless $whos_turn;
   $chosen_card++;
   my @hand = sorted_hand($whos_turn);
   $chosen_card = 0 if $chosen_card > $#hand;
   if($lead_suit && scalar(grep { substr($_,0,1) eq $lead_suit } @hand) &&  # if we have to follow suit
      substr($hand[$chosen_card],0,1) ne $lead_suit) {  # and we were selecting the wrong suit
      $chosen_card = 0;  # start with first card
      while(substr($hand[$chosen_card],0,1) ne $lead_suit) {
        $chosen_card++;
      }
   }
   &{$exposures{$whos_turn}};
};

my %card_left_x;  # $card_left_x{$dir}{$card} = $x  #  for decoding mouse clicks 

sub expose_drawingarea_north
{
   # North hand
   my(undef,undef,$width,$height) = $da_north->window->get_geometry();  # get size of drawing area
   $da_north->window->clear_area(0,0,$width,$height);
   $card_left_x{'N'} = {};  # no cards
   my @hand = sorted_hand('N',$trick > 13);
   my $xstart = ($width/2) - ($#hand+1) *10 - 40;  # horizontally center cards
   if($show_hands =~ /N/ || $trick > 13 || $declarer eq "S") {
      foreach my $index (0..$#hand) {  # for each card in the south hand
         my $suitx = $card::suithash{substr($hand[$index],0,1)} * 15;  # gap inbetween suits
         card::draw($da_north,$hand[$index],$xstart+$index*20+$suitx,6,64,$chosen_card == $index && $whos_turn eq 'N');
         $card_left_x{'N'}{$hand[$index]} = $xstart+$index*20+$suitx;
      }
   } else {  # else hide hand
      foreach my $index (0..$#hand) {  # for each card in the south hand
         card::draw($da_north,'BACK',$xstart+$index*20,6,64);
      }
   }
}
sub expose_drawingarea_south
{
   # South hand
   my(undef,undef,$width,$height) = $da_south->window->get_geometry();  # get size of drawing area
   $da_south->window->clear_area(0,0,$width,$height);
   $card_left_x{'S'} = {};    # no cards
   my @hand = sorted_hand('S',$trick > 13);
   my $xstart = ($width/2) - ($#hand+1) *10 - 40;  # horizontally center cards
   foreach my $index (0..$#hand) {  # for each card in the north hand
      my $suitx = $card::suithash{substr($hand[$index],0,1)} * 15;  # gap inbetween suits
      card::draw($da_south,$hand[$index],$xstart+$index*20+$suitx,6,64,$chosen_card == $index && $whos_turn eq 'S');
      $card_left_x{'S'}{$hand[$index]} = $xstart+$index*20+$suitx;
   }
}
sub expose_drawingarea_west
{
   # West hand
   my(undef,undef,$width,$height) = $da_west->window->get_geometry();  # get size of drawing area
   $da_west->window->clear_area(0,0,$width,$height);
   my($rowsuit,$num_in_row,$num_drawn_in_row,$num_rows,$spacing) = ("x",0,0,0,0);
   my @hand = sorted_hand('W',$trick > 13);
   if($show_hands =~ /W/ || $trick > 13 || ($declarer eq "E" && "$trick$whos_turn" ne "1S")) {
      foreach my $index (0..$#hand) {  # for each card in the west hand
         if(substr($hand[$index],0,1) ne $rowsuit) {
            $rowsuit = substr($hand[$index],0,1);
            $num_in_row = grep { substr($_,0,1) eq $rowsuit } @hand;
            $num_drawn_in_row = 0;
            $spacing = $width/$num_in_row if $num_in_row;
            $spacing = 20 if $spacing > 20;
            $num_rows++;
         };
         my $suitx = $card::suithash{substr($hand[$index],0,1)} * 15;  # gap inbetween suits
         card::draw($da_west,$hand[$index],$num_drawn_in_row*$spacing,50+$num_rows*50,64);
         $num_drawn_in_row++;
      }
   } else {  # else hide hand
      foreach my $index (0..$#hand) {  # for each card in the west hand
         card::draw($da_west,'BACK',6,50+ (12-$#hand)*10 + $index*20,64);
      }
   }
}
sub expose_drawingarea_east
{
   # East hand
   my(undef,undef,$width,$height) = $da_east->window->get_geometry();  # get size of drawing area
   $da_east->window->clear_area(0,0,$width,$height);
   my($rowsuit,$num_in_row,$num_drawn_in_row,$num_rows,$spacing,$left_space) = ("x",0,0,0,0,0);
   my @hand = sorted_hand('E',$trick > 13);
   if($show_hands =~ /E/ || $trick > 13 || $declarer eq "W") {
      foreach my $index (0..$#hand) {  # for each card in the east hand
         if(substr($hand[$index],0,1) ne $rowsuit) {
            $rowsuit = substr($hand[$index],0,1);
            $num_in_row = grep { substr($_,0,1) eq $rowsuit } @hand;
            $num_drawn_in_row = 0;
            $spacing = $width/$num_in_row if $num_in_row;
            $spacing = 20 if $spacing > 20;
            $left_space = $width - 60 - 20 * $num_in_row;
            $left_space = 0 if $left_space < 0;
            $num_rows++;
         };
         $num_in_row--;
         my $suitx = $card::suithash{substr($hand[$index],0,1)} * 15;  # gap inbetween suits
         card::draw($da_east,$hand[$index],$left_space+$num_drawn_in_row*$spacing,50+$num_rows*50,64);
         $num_drawn_in_row++;
      }
   } else {  # else hide hand
      foreach my $index (0..$#hand) {  # for each card in the west hand
         card::draw($da_east,'BACK',56,50+ (12-$#hand)*10 + $index*20,64);
      }
   }
}
sub expose_drawingarea_play
{
   my(undef,undef,$width,$height) = $da_play->window->get_geometry();  # get size of drawing area
   $da_play->window->clear_area(0,0,$width,$height);
   return if ! $trick || $trick > 13;
   card::draw($da_play,played("N$trick"),$width/2 - 35,$height/7  ,undef,$lead[$trick+1] eq "N") if played("N$trick"); 
   card::draw($da_play,played("W$trick"),$width/2 - 75,2*$height/7,undef,$lead[$trick+1] eq "W") if played("W$trick");
   card::draw($da_play,played("E$trick"),$width/2 + 5,2*$height/7 ,undef,$lead[$trick+1] eq "E") if played("E$trick");
   card::draw($da_play,played("S$trick"),$width/2 - 35,3*$height/7,undef,$lead[$trick+1] eq "S") if played("S$trick");
   card::arrow($da_play,'S',$width/2-15,$height-20) if $whos_turn eq 'S';
   card::arrow($da_play,'N',$width/2-15,0) if $whos_turn eq 'N';
}
sub expose_drawingareas
{
   expose_drawingarea_south();
   expose_drawingarea_north();
   expose_drawingarea_play();
   expose_drawingarea_west();
   expose_drawingarea_east();
}

sub show_center
{
   my($center_name) = @_;
   return if $curr_center eq $center_name;
   $center_widgets{$curr_center}->hide();
   $center_widgets{$center_name}->show();
   $curr_center = $center_name;
}

sub do_exit{ exit 0 } 
sub do_about { require about; about::open_window(); }
sub do_nothing{  } 

sub do_key_press_event 
{ 
   my($widget,$event) = @_;

   #print map { "   $_ -> " . $$event{$_} ."\n"} keys %$event;
   if($curr_center eq "play") {
      play_key_left() if $$event{'keyval'} == 65361;
      play_key_right() if $$event{'keyval'} == 65363;
   }
   return unless defined $$event{'string'};  # ignore events without key code (e.g. Alt key)
   my $func = ${$center_keystrokes{$curr_center}}{$$event{'string'}};
   return unless $func;
   &$func(@_);
   return undef;
}

sub end_trick_afterglow
{
   return if $whos_turn ne "" || $trick > 13 || $trick < 1;
   $trick++;
   $lead_suit = "";
   if($trick > 13) {  # if this is not a trick
      score::score();
      expose_drawingareas();
   } else {  # else we are about to play a trick
      $whos_turn = $lead[$trick];
      $card_click_this_trick[$trick] = 0;  
      expose_drawingarea_play();
   }
   play_ai_while_we_can();
}

sub do_da_click   # drawing area
{ 
   my($widget,$which,$event) = @_;
   #print "do_da_click() which=$which button=$$event{'button'} x=$$event{'x'} y=$$event{'y'}\n"  if $debug; 
   #print "do_da_click() ". join(" - ",@_) ."\n" if $debug; 
   #print map { "   $_ => ". ${$event}{$_} ."\n" } sort keys %$event;
   if($curr_center eq "play") {  # only if we are in play mode
      if($whos_turn eq "") {  # if end of trick "afterglow"
         end_trick_afterglow();
      } elsif($which eq $whos_turn) {  # else we are picking a card, so if they clicked the hand whos turn it is
         $card_click_this_trick[$trick] = 1;
         my($bestx,$card) = (0,"");
         foreach (keys %{$card_left_x{$which}}) {  # for every card drawn on screen
            my $cardx = $card_left_x{$which}{$_};  # determin left edge of this card on screen
            if($$event{'x'} > $cardx && $$event{'x'} < $cardx + 80 && $cardx > $bestx){  # if match and best (most right, pun intended)
                ($bestx,$card) = ($cardx,$_);
            };
         }
         $card = "" if($lead_suit && scalar(grep { substr($_,0,1) eq $lead_suit } sorted_hand($whos_turn)) && #if we have to follow suit
                       substr($card,0,1) ne $lead_suit);  # and we were selecting the wrong suit
         play_card($card) if $card;
         play_ai_while_we_can();
      }
   };
   return undef;
}

sub do_play_spacebar 
{ 
   if($whos_turn eq "") {
      end_trick_afterglow();
   } else {
      my @hand = sorted_hand($whos_turn);
      $card_click_this_trick[$trick] = 1;
      play_card($hand[$chosen_card]) if($chosen_card >= 0 && $chosen_card <= $#hand);
      play_ai_while_we_can();
   }
}

sub build_menu_item
{
   my($parent,$title,$accelgroup,$key,$handler,$disabled) = @_;

   my $menuitem = new Gtk::MenuItem($title);
   $menuitem->child->parse_uline($title);
   if($key) {
      $menuitem->add_accelerator('activate_item', $accelgroup, unpack("C",$1), 'mod1_mask', ['visible', 'locked'] ) if $key =~ s/Alt\+(.)//;
      $menuitem->add_accelerator('activate_item', $accelgroup, unpack("C",$1), 'control_mask', ['visible', 'locked'] ) if $key =~ s/Ctrl\+(.)//;
      $menuitem->add_accelerator('activate_item', $accelgroup, 0xFFBD + $1, 0, ['visible', 'locked'] ) if $key =~ s/F(\d+)//;
      $menuitem->add_accelerator('activate_item', $accelgroup, unpack("C",$key), 0, ['visible', 'locked'] ) if $key;
   };
   $menuitem->signal_connect('activate', $handler, '', 'quit1', "window1-instance" ) if $handler;
   $parent->append($menuitem);
   $menuitem->set_sensitive(0) if $disabled;
   $menuitem->show();
   return $menuitem;
}

sub build_mainwindow
{
   $main_window = new Gtk::Window( 'toplevel' );
   $main_window->title("OpenBridge");
   $main_window->set_policy(0,1,0);  # no allow_shrink, yes allow_grow, no auto_shrink
   #$main_window->set_default_size(600,400);
   $main_window->signal_connect('delete_event', \&do_exit ) or die; # quit on main window close
   $main_window->signal_connect('key_press_event', \&do_key_press_event ) or die; 
   #$main_window->border_width(0);
   my $vbox = new Gtk::VBox( 0, 0 ); # non-homogenous, zero padding
   $main_window->add($vbox);  
   # keyboard accelerator
   my $accelgroup = new Gtk::AccelGroup;
   $accelgroup->attach($main_window);
   # menu bar
   my $menubar = new Gtk::MenuBar;
   $menubar->set_shadow_type('out' );
   $vbox->pack_start($menubar,0,0,0); # child, no expand, no fill, zero padding
   # Game menu
   my $game_menuitem = build_menu_item($menubar,"_Game",$accelgroup,"Alt+G");
   my $game_menu = new Gtk::Menu;
   $game_menuitem->set_submenu($game_menu);
   build_menu_item($game_menu,"_Bid Next Deal..",$accelgroup,"Alt+N",\&deal::new_deal_now);
   build_menu_item($game_menu,"New _Deal...",$accelgroup,"Alt+D",\&deal::interactive_deal);
   build_menu_item($game_menu,"_Auto Move",$accelgroup,"Alt+A",\&do_auto);
   $game_menu->append(my $separator1 = Gtk::MenuItem->new());  # separator
   $separator1->set_sensitive(0); # This matters for keyboard
   build_menu_item($game_menu,"_Save",$accelgroup,"Ctrl+S",\&pbn::do_save);
   $game_menu->append(my $separator4 = Gtk::MenuItem->new());  # separator
   $separator4->set_sensitive(0); # This matters for keyboard
   build_menu_item($game_menu,"E_xit",$accelgroup,"Alt+Q",\&do_exit);
   # Edit menu
   my $edit_menuitem = build_menu_item($menubar,"_Edit",$accelgroup,"Alt+E");
   my $edit_menu = new Gtk::Menu;
   $edit_menuitem->set_submenu($edit_menu);
   build_menu_item($edit_menu,"_Cut",$accelgroup,"Ctrl+X",\&do_nothing,1);
   build_menu_item($edit_menu,"C_opy",$accelgroup,"Ctrl+C",\&do_nothing,1);
   build_menu_item($edit_menu,"_Paste",$accelgroup,"Ctrl+P",\&do_nothing,1);
   $edit_menu->append(my $separator2 = Gtk::MenuItem->new());  # separator
   $separator2->set_sensitive(0);
   build_menu_item($edit_menu,"_Redo",$accelgroup,"",\&do_redo,1);
   build_menu_item($edit_menu,"_Undo",$accelgroup,"Ctrl+Z",\&do_undo);
   # Show menu
   my $show_menuitem = build_menu_item($menubar,"_Show",$accelgroup,"Alt+S");
   my $show_menu = new Gtk::Menu;
   $show_menuitem->set_submenu($show_menu);
   build_menu_item($show_menu,"_Hint",$accelgroup,"Alt+H",\&do_hint);
   build_menu_item($show_menu,"History of _Play",$accelgroup,"F9",\&history::open_window);
   $show_menu->append(my $separator3 = Gtk::MenuItem->new());  # separator
   $separator3->set_sensitive(0);
   build_menu_item($show_menu,"_My Hand Only",$accelgroup,"Alt+M",\&deal::show_mine_clicked);
   build_menu_item($show_menu,"_North Hand",$accelgroup,"F6",\&deal::show_north_clicked);
   build_menu_item($show_menu,"_East & West Hands",$accelgroup,"F7",\&deal::show_ew_clicked);
   build_menu_item($show_menu,"_All Hands",$accelgroup,"F5",\&deal::show_all_clicked);
   # Help menu
   my $help_menuitem = build_menu_item($menubar,"Help");
   my $help_menu = new Gtk::Menu;
   $help_menuitem->set_submenu($help_menu);
   build_menu_item($help_menu,"_About",$accelgroup,"F1",\&do_about);
   # drawing area shenanigans
   my $hbox2 = new Gtk::HBox( 0, 0 ); # non-homogenous, zero padding
   $vbox->pack_start($hbox2,1,1,0); # child, yes expand, yes fill, zero padding
   # drawing area west
   $da_west = Gtk::DrawingArea->new();
   $da_west->size(140,400);
   $da_west->signal_connect('expose_event', \&expose_drawingarea_west ); 
   $hbox2->pack_start($da_west,1,1,0); # child, yes expand, yes fill, zero padding
   $da_west->add_events("button-press-mask");
   $da_west->signal_connect('button_press_event', \&do_da_click, "W" ) or die; 
   # drawing area east
   $da_east = Gtk::DrawingArea->new();
   $da_east->size(140,400);
   $da_east->signal_connect('expose_event', \&expose_drawingarea_east ); 
   $hbox2->pack_end($da_east,1,1,0); # child, yes expand, yes fill, zero padding
   $da_east->add_events("button-press-mask");
   $da_east->signal_connect('button_press_event', \&do_da_click, "E" ) or die; 
   # drawing area north
   my $vbox2 = new Gtk::VBox( 0, 0 ); # non-homogenous, zero padding
   $hbox2->pack_start($vbox2,1,1,0); # child, yes expand, yes fill, zero padding
   $da_north = Gtk::DrawingArea->new();
   $da_north->size(400,70);
   $da_north->signal_connect('expose_event', \&expose_drawingarea_north ); 
   $vbox2->pack_start($da_north,1,1,0); # child, yes expand, yes fill, zero padding
   $da_north->add_events("button-press-mask");
   $da_north->signal_connect('button_press_event', \&do_da_click, "N" ) or die; 
   # drawing area play (center)
   $da_play = Gtk::DrawingArea->new();  # in the middle
   $da_play->size(400,300);
   $da_play->signal_connect('expose_event', \&expose_drawingarea_play ); 
   $vbox2->pack_start($da_play,1,1,0); # child, yes expand, yes fill, zero padding
   $da_play->add_events("button-press-mask");
   $da_play->signal_connect('button_press_event', \&do_da_click, "play" ) or die; 
   # drawing area south
   $da_south = Gtk::DrawingArea->new();
   $da_south->size(400,70);
   $da_south->signal_connect('expose_event', \&expose_drawingarea_south ); 
   my $south_align = Gtk::Alignment->new(1,1,0,0);  # bottom alignment, 0% expansion
   $south_align->add($da_south);
   $vbox2->pack_end($south_align,1,1,0); # child, yes expand, yes fill, zero padding
   $da_south->add_events("button-press-mask");
   $da_south->signal_connect('button_press_event', \&do_da_click, "S" ) or die; 
   # separator for status bar
   my $hseparator = new Gtk::HSeparator(); 
   $vbox->pack_start($hseparator,0,0,0); # child, no expand, no fill, zero padding
   # hbox (of status bar)
   my $hbox = new Gtk::HBox( 0, 0 ); # non-homogenous, zero padding
   $vbox->pack_start($hbox,0,0,0); # child, no expand, no fill, zero padding
   # status bar
   $status_label = Gtk::Label->new("Welcome to OpenBridge");
   $hbox->pack_start($status_label,1,0,3); # child, yes expand, no fill, 3 padding
   $hbox->pack_start(Gtk::VSeparator->new(),0,0,0); # child, no expand, no fill, zero padding
   $dealer_label = Gtk::Label->new("Dealer: $dealer");
   $hbox->pack_start($dealer_label,0,0,3); # child, no expand, no fill, zero padding
   $hbox->pack_start(Gtk::VSeparator->new(),0,0,0); # child, no expand, no fill, zero padding
   $contract_label = Gtk::Label->new("Contract: --");
   $hbox->pack_start($contract_label,0,0,3); # child, no expand, no fill, zero padding
   $hbox->pack_start(Gtk::VSeparator->new(),0,0,0); # child, no expand, no fill, zero padding
   $nswins_label = Gtk::Label->new("N/S Wins: --");
   $hbox->pack_start($nswins_label,0,0,3); # child, no expand, no fill, zero padding
   $hbox->pack_start(Gtk::VSeparator->new(),0,0,0); # child, no expand, no fill, zero padding
   $ewwins_label = Gtk::Label->new("E/W Wins: --");
   $hbox->pack_start($ewwins_label,0,0,3); # child, no expand, no fill, zero padding
   # show me the money!
   $main_window->show_all();
   # center switchable
   $center_widgets{'play'} = $da_play;
   $center_keystrokes{'play'} = { " " => \&do_play_spacebar, "\n" => \&do_play_spacebar, 
                                 u => \&do_undo, a => \&do_auto, h => \&do_hint };  
   $curr_center = "play";
   # dealing
   ($center_widgets{'deal'},$center_keystrokes{'deal'}) = deal::create_widgets();  # returns list
   $center_widgets{'deal'}->hide();
   $vbox2->pack_start($center_widgets{'deal'},1,1,0); # child, yes expand, yes fill, zero padding
   # bidding
   ($center_widgets{'bid'},$center_keystrokes{'bid'}) = bid::create_widgets();  # returns list
   $center_widgets{'bid'}->hide();
   $vbox2->pack_start($center_widgets{'bid'},1,1,0); # child, yes expand, yes fill, zero padding
   # scoring
   ($center_widgets{'score'},$center_keystrokes{'score'}) = score::create_widgets();  # returns list
   $center_widgets{'score'}->hide();
   $vbox2->pack_start($center_widgets{'score'},1,1,0); # child, yes expand, yes fill, zero padding
}

card::load_deck("bonded.deck");
build_mainwindow();
deal::new_deal_now();

Gtk->main();

exit (0);
