/*
 * radeonlcd.c
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * Written by Sos Pter <sp@osb.hu>, 2004
 */

#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/pm.h>

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
#include "radeon.h"
#else
#include <video/radeon.h>
#include <linux/device.h>
#endif

#include <asm/io.h>

#include "radeonlcd.h"

#define RADEONLCD_VERSION "2004-02-04"

#define MODULE_NAME "radeonlcd"

enum radeon_chips {
        RADEON_LW,
        RADEON_LX,
        RADEON_LY,
        RADEON_LZ,
        RADEON_Ld,
        RADEON_Le,
        RADEON_Lf,
        RADEON_Lg
};

static struct pci_device_id radeonlcd_pci_table[] = {
	{ PCI_VENDOR_ID_ATI, PCI_DEVICE_ID_ATI_RADEON_LW, PCI_ANY_ID, PCI_ANY_ID, 0, 0, RADEON_LW},
	{ PCI_VENDOR_ID_ATI, PCI_DEVICE_ID_ATI_RADEON_LX, PCI_ANY_ID, PCI_ANY_ID, 0, 0, RADEON_LX},
	{ PCI_VENDOR_ID_ATI, PCI_DEVICE_ID_ATI_RADEON_LY, PCI_ANY_ID, PCI_ANY_ID, 0, 0, RADEON_LY},
	{ PCI_VENDOR_ID_ATI, PCI_DEVICE_ID_ATI_RADEON_LZ, PCI_ANY_ID, PCI_ANY_ID, 0, 0, RADEON_LZ},
	{ PCI_VENDOR_ID_ATI, PCI_DEVICE_ID_ATI_RADEON_Ld, PCI_ANY_ID, PCI_ANY_ID, 0, 0, RADEON_Ld},
	{ PCI_VENDOR_ID_ATI, PCI_DEVICE_ID_ATI_RADEON_Le, PCI_ANY_ID, PCI_ANY_ID, 0, 0, RADEON_Le},
	{ PCI_VENDOR_ID_ATI, PCI_DEVICE_ID_ATI_RADEON_Lf, PCI_ANY_ID, PCI_ANY_ID, 0, 0, RADEON_Lf},
	{ PCI_VENDOR_ID_ATI, PCI_DEVICE_ID_ATI_RADEON_Lg, PCI_ANY_ID, PCI_ANY_ID, 0, 0, RADEON_Lg},
	{ 0, }
};
MODULE_DEVICE_TABLE(pci, radeonlcd_pci_table);

static unsigned long mmio_base;

#ifdef CONFIG_VT
extern int (*console_blank_hook)(int);
#endif

static void radeon_mem_read(u32 addr, u32 *data)
{
	char *base = ioremap(addr, 1);

	*data = readb(base);
	iounmap(base);
}

static void radeon_mem_write(u32 addr, u32 data)
{
	char *base = ioremap(addr, 1);

	writeb(data, base);
	iounmap(base);
}

int radeonlcd_blank(int blank)
{
	u32 lvds;

	radeon_mem_read(mmio_base + LVDS_GEN_CNTL, &lvds);
	lvds = blank ? lvds & ~LVDS_ON : lvds | LVDS_ON;
	radeon_mem_write(mmio_base + LVDS_GEN_CNTL, lvds);

	return 0;
}	
EXPORT_SYMBOL(radeonlcd_blank);

static int radeonlcd_register(struct pci_dev *pdev, const struct pci_device_id *ent)
{
	/* Enable device in PCI config */
	if (pci_enable_device(pdev) != 0) {
		printk(KERN_ERR "%s: Cannot enable PCI device\n", MODULE_NAME);
		return -ENODEV;
	}

	mmio_base = pci_resource_start(pdev, 2);

	printk(KERN_INFO "%s version: %s\n", MODULE_NAME, RADEONLCD_VERSION);
#ifdef CONFIG_PCI_NAMES
	printk(KERN_INFO "%s: %s detected.\n", MODULE_NAME, pdev->pretty_name);
#endif
	printk(KERN_INFO "%s: MMIO registers at 0x%lx.\n", MODULE_NAME, mmio_base);

#ifdef CONFIG_VT
	if (console_blank_hook == NULL) {
		console_blank_hook = radeonlcd_blank;
		printk(KERN_INFO "%s: LCD backlight turn off at console blanking is enabled.\n", MODULE_NAME);
	}
#endif
	return 0;
}

static void radeonlcd_unregister (struct pci_dev *pdev)
{
#ifdef CONFIG_VT
	if (console_blank_hook == radeonlcd_blank) {
		console_blank_hook = NULL;
	}
#endif
}

static struct pci_driver radeonlcd_driver = {
	.name		= MODULE_NAME,
	.id_table	= radeonlcd_pci_table,
	.probe		= radeonlcd_register,
	.remove		= radeonlcd_unregister
};


static int __init radeonlcd_init(void)
{
	return pci_module_init(&radeonlcd_driver);
}

static void __exit radeonlcd_exit(void)
{
	pci_unregister_driver(&radeonlcd_driver);
}

module_init(radeonlcd_init);
module_exit(radeonlcd_exit);

MODULE_AUTHOR("Sos Pter <sp@osb.hu>");
MODULE_DESCRIPTION("LCD backlight handler for laptops equipped with mobile ATI Radeon chips.");
MODULE_LICENSE("GPL");

/* End of file */
