#!/usr/bin/perl -w
# -*- Mode: perl; indent-tabs-mode: nil -*-
#
# The contents of this file are subject to the Mozilla Public
# License Version 1.1 (the "License"); you may not use this file
# except in compliance with the License. You may obtain a copy of
# the License at http://www.mozilla.org/MPL/
#
# Software distributed under the License is distributed on an "AS
# IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
# implied. See the License for the specific language governing
# rights and limitations under the License.
#
# The Original Code is mozilla.org code.
#
# The Initial Developer of the Original Code is Holger
# Schurig. Portions created by Holger Schurig are
# Copyright (C) 1999 Holger Schurig. All
# Rights Reserved.
#
# Contributor(s): Holger Schurig <holgerschurig@nikocity.de>
#                 Terry Weissman <terry@mozilla.org>
#                 Dan Mosedale <dmose@mozilla.org>
#                 Dave Miller <justdave@syndicomm.com>
#                 Frederick Dean <software@fdd.com>
#
#
#
#
# Hey, what's this?
#
# 'checksetup.pl' is a script that is supposed to run during installation
# time and also after every upgrade.
#
# The goal of this script is to make the installation even more easy.
# It does so by doing things for you as well as testing for problems
# early.
#
# And you can re-run it whenever you want. Especially after DCVS
# gets updated you SHOULD rerun it. Because then it may update your
# SQL table definitions so that they are again in sync with the code.
#
# So, currently this module does:
#
#     - check for required perl modules
#     - set defaults for local configuration variables
#     - create and populate the data directory after installation
#     - set the proper rights for the *.cgi, *.html ... etc files
#     - check if the code can access MySQL
#     - creates the database 'bugs' if the database does not exist
#     - creates the tables inside the database if they don't exist
#     - automatically changes the table definitions of older BugZilla
#       installations
#     - populates the groups
#     - put the first user into all groups so that the system can
#       be administrated
#     - changes already existing SQL tables if you change your local
#       settings, e.g. when you add a new platform
#
# People that install this module locally are not supposed to modify
# this script. This is done by shifting the user settable stuff into
# a local configuration file 'config.pm'. When this file get's
# changed and 'checkconfig.pl' will be re-run, then the user changes
# will be reflected back into the database.
#
# Developers however have to modify this file at various places. To
# make this easier, I have added some special comments that one can
# search for.
#
#     To                                               Search for
#
#     add/delete local configuration variables         --LOCAL--
#     check for more prerequired modules               --MODULES--
#     change the defaults for local configuration vars --LOCAL--
#     update the assigned file permissions             --CHMOD--
#     add more MySQL-related checks                    --MYSQL--
#     change table definitions                         --TABLE--
#     add more groups                                  --GROUPS--
#     create initial administrator account            --ADMIN--
#
# Note: sometimes those special comments occur more then once. For
# example, --LOCAL-- is at least 3 times in this code!  --TABLE--
# also is used more than once. So search for every occurence!
#


use diagnostics;
use strict;

###########################################################################
# Check and update local configuration
###########################################################################

#
# This is quite tricky. But fun!
#
# First we read the file 'config.pm'. Then we check if the variables we
# need are defined. If not, config.pm will be amended by the new settings
# and the user informed to check this. The program then stops.
#
# Why do it this way around?
#
# Assume we will enhance DCVS and eventually more local configuration
# stuff arises on the horizon.
#
# But the file 'config.pm' is not in the DCVS archive or tarfile. You
# know, we never want to overwrite your own version of 'config.pm', so
# we can't put it into the archive/tarfile, can we?
#
# Now, we need a new variable. We simply add the necessary stuff to checksetup.
# The user get's the new version of DCVS from the archive, runs checksetup
# and checksetup finds out "Oh, there is something new". Then it adds some
# default value to the user's local setup and informs the user to check that
# to see if that is what the user wants.
#
# Cute, ey?
#

if(-e "config.pm") {
   if(! -r "config.pm") {
      die "unix file permissions problem with config.pm. I cannot read it.\n";
   } else {
      do 'config.pm' or die("syntax error in config.pm $@");
   }
}

my $newstuff = "";
sub LocalVar ($$)
{
    my ($name, $definition) = @_;
    return if ($config::{$name}); # if localconfig declared it, we're done.
    $newstuff .= "\n   \$config::" . $name;
    if(! -e "config.pm") {
       open FILE, '>config.pm' or die "\n\tI cannot open config.pm\n\n";
       print(FILE "#!/usr/bin/perl -w -T\n".
                  "use strict;\n".
                  "package config;\n".
                  "\n".
                  "# Don't check this file into CVS.\n".
                  "# It is generated (grown) by checkconfig.pl.\n".
                  "\n");
    }
    open FILE, '>>config.pm' or die "\n\tI cannot open config.pm\n\n";
    print FILE $definition;
    close FILE;
}

#
# Set up the defaults for the --LOCAL-- variables below:
#
    

my $htmlPath = libPath();
$htmlPath =~ s/src.?$/html/;
LocalVar('rootdir', '
#
#  This is the directory which the path after the wildcat.cgi 
#  maps to.
#
our $rootdir = "'. $htmlPath .'";
');

LocalVar('debug', '
#
# This should not be enabled on a live system.
# This is a flag, set it to undef to turn it off.
our $debug = 0;
');

LocalVar('scriptUser', '
#
#  The config.pm contains the database password for wildcat, so we need to protect it.
#  Anyone with that password roots version control.
#  Our plan makes config.pm only readable by $scriptUser, and makes
#  wildcat.cgi SUID $scriptUser.  
#
#  Better strategies exist where the scripts run without privileges to write
#  to the scripts (i.e. also not ownership).
#  Instead, they use group permission to read the password.  
#  A chroot jail would also help (athough lookout for load and save of RCS files).
#  If you do not want checksetup.pl to do file permissions, set $scriptUser to undef.
#
#  WARNING: If you use the userid of your web server, anyone who can write a
#  cgi (or exploit a cgi hole) can view the database password.
our $scriptUser = "wildcat";
');

LocalVar('passwd', '
#
# To make changes someone must know this pass prase.
our $passwd = undef;
');

LocalVar('copyright', '
#
# included in footer
our $copyright = "Copyright &copy; me 2002";
');

LocalVar('timezone', '
#
# What timezone should the web pages report in...
our $timezone = "CST";
');

LocalVar('timeformat', '
#
# What timeformat should the web pages report with?
# Run "perldoc Date::Format" for the meaning of escape codes.
our $timeformat = "%a %b %d, %Y %H:%M:%S %Z";
');

LocalVar('sizes', '
#
# Number of pixels per image.
# Thumbnails are only made if the thumbnail number of pixes is less than 80% or orig
our $sizes =     [ 2000,  20000,200000, 600000,  2500000,    2000000000 ];
');

LocalVar('sizenames', '
our $sizenames = [ "icon","thumb","half-screen","screen","oversized","full-orig"];
');

LocalVar('idleTimeout', '
#  Login Idle Timeout in seconds
our $idleTimeout = 3600;
');


if ($newstuff ne "") {
    print "\n".
          "This version of wildcat contains some variables that you may\n".
          "want to adapt to your local settings...\n".
          "$newstuff\n".
          "\n".
          "Please edit the file 'config.pm' and rerun checksetup.pl\n\n";
    exit;
}

# 2000-Dec-18 - justdave@syndicomm.com - see Bug 52921
# This is a hack to read in the values defined in config.pm without getting
# them predeclared at compile time if they're missing from config.pm.
# Ideas swiped from pp. 281-282, O'Reilly's "Programming Perl 2nd Edition"
# Note that we won't need to do this in globals.pl because globals.pl couldn't
# care less whether they were defined ahead of time or not. 
my $my_scriptUser = ${*{$config::{'scriptUser'}}{SCALAR}};
my $my_rootdir = ${*{$config::{'rootdir'}}{SCALAR}};

###########################################################################
# Update the @INC path in the scripts
###########################################################################

use Cwd;

sub libPath {
   $0 =~ /(.*)\/[^\/]+$/;  # match path before executable
   my $libPath = $1;
   if($libPath !~ /^\//) { # if path doesn't begin with slash
      $libPath = Cwd::abs_path($libPath);
   }
   return $libPath;
}
my $libPath = libPath();

print("\n");
my @scripts = qw( wildcat.cgi );
push @scripts, glob("$libPath/test/*.pl");  # do perl scripts in test directory
for my $file (@scripts) {
   $file = "$libPath/$file" unless $file =~ /^\//;  # unless begins with slash
   open(INPL,"<$file") or die("could not open required file $file");
   my $numChars = read(INPL,my $filedata,10000); 
   if($filedata =~ s/^(\s*BEGIN\s*\{\s*(?:push|unshift)[\(\s]+\@INC\s*,\s*")(.*)"/$1$libPath"/m) {
      next if($2 eq $libPath);  # if old value is same as current
   } elsif($filedata =~ s/^(\s*use\s+lib)\s+([^;]+)/$1 "$libPath"/m) {
      next if($2 eq "\"$libPath\"");  # if old value is same as current
   } else {
      die("\nI couldn't find a line like ...\n".
             "   use lib \"/path/to/libs\";\n".
             "or... \n".
             "   BEGIN { push \@INC, \"/path/to/libs\" };\n".
             "...in file $file to update\n");
   };
   print("Updating \@INC path of $file\n");
   open(OUTPL,">$file.$$") or die;
   while(1) {
      defined($numChars) or die;
      last if $numChars == 0;
      print(OUTPL $filedata); 
      $numChars = read(INPL,$filedata,10000); 
   }
   close(INPL) or die;
   close(OUTPL) or die;
   rename("$file.$$","$file") or die; 
}
close(INPL);
close(OUTPL);

###########################################################################
# File permissions
###########################################################################

sub adduserCmdName {
   for my $name qw(/sbin/adduser /usr/sbin/adduser /sbin/useradd /usr/sbin/useradd) {
      return $name if -e $name;
   }
}

my $rootBitch = "";

sub fixOwnership {
   my($username,@filenames) = (@_);

   my $uid = $username;
   if($uid !~ /^\d+$/) {  # if username is not all numeric
      $uid   = getpwnam($username);
      if(! defined $uid) {
         $rootBitch .= "\t". adduserCmdName() ." $my_scriptUser";
         $uid = -1;
      };
   };
   foreach my $filename (@filenames) {
      my $group = (stat($filename))[5];  # get group so we can leave it unchanged
      chown($uid,$group,$filename);  # okay if fails
      my $file_uid = (stat($filename))[4];   
      if($uid != $file_uid) {
         $rootBitch .= "\tchown $username $filename\n";
      }
   };
}

sub fixMode {
   my($mode,@filenames) = (@_);

   foreach my $filename (@filenames) {
      chmod $mode, $filename;
      my $actualMode = (stat($filename))[2] & 07777;
      if($actualMode != $mode) {
         $rootBitch .= sprintf("\tchmod %o $filename\n",$mode);
      }
   };
}

if(defined $my_scriptUser) {
   fixOwnership($my_scriptUser,"config.pm","wildcat.cgi",$my_rootdir);
   fixMode( 0600, "config.pm");    # -rw-------
   fixMode( 0755, @scripts);       # -rwxr-xr-x
   fixMode(04755, "wildcat.cgi");  # -rwsr-xr-x
   fixMode( 0755, ".");            # -rwxr-xr-x
}

if($rootBitch) {
   print("There are some commands which need to be run as root.\n".
         "You can either run them yourself or rerun this script as\n".
         "root and I will do them.\n");
   print($rootBitch);
   exit(-1);
}


#
# Final checks...

print("\n".
      "Well, then.  Everything looks great.\n".
      "You may still need to configure Apache \n".
      "to run wildcat.cgi.  For example...\n".
      "  ScriptAlias /wildcat $libPath/wildcat.cgi\n\n");




